import Foundation

public struct SVG: HTML, Sendable {
    private enum Storage {
        case raw(String)
        case base64(String)
    }

    let description: String
    private let storage: Storage

    public init(_ description: String, content: () -> String) {
        self.storage = .raw(content())
        self.description = description
    }

    public init(base64: String, description: String) {
        self.storage = .base64(base64)
        self.description = description
    }

    public var body: some HTML {
        HTMLGroup {
            switch storage {
            case let .raw(raw):
                HTMLRaw(raw)
            case let .base64(base64):
                img(src: "data:image/svg+xml;base64,\(base64)", alt: "\(self.description)")
            }
        }
    }
}

extension SVG {
    public static let error = Self("Error") {
    """
    <?xml version="1.0" encoding="UTF-8"?>
    <!--Generator: Apple Native CoreSVG 326-->
    <!DOCTYPE svg
    PUBLIC "-//W3C//DTD SVG 1.1//EN"
           "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
    <svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 13.1641 12.2266">
     <g>
      <rect height="12.2266" opacity="0" width="13.1641" x="0" y="0"/>
      <path d="M9.78906 0.640625L12.2266 3.35156C12.6484 3.82031 12.7578 4.125 12.7578 4.74219L12.7578 7.48438C12.7578 8.10156 12.6484 8.40625 12.2266 8.875L9.78906 11.5859C9.40625 12 9.01562 12.2266 8.30469 12.2266L4.44531 12.2266C3.74219 12.2266 3.34375 12 2.96875 11.5859L0.523438 8.875C0.109375 8.40625 0 8.10156 0 7.48438L0 4.74219C0 4.125 0.109375 3.82031 0.523438 3.35156L2.96875 0.640625C3.34375 0.226562 3.74219 0 4.44531 0L8.30469 0C9.01562 0 9.40625 0.226562 9.78906 0.640625ZM8.05469 3.64844L6.375 5.3125L4.71094 3.65625C4.60938 3.55469 4.48438 3.5 4.32031 3.5C4.00781 3.5 3.75 3.74219 3.75 4.07031C3.75 4.21875 3.8125 4.35938 3.91406 4.46875L5.5625 6.11719L3.91406 7.76562C3.8125 7.875 3.75 8.01562 3.75 8.16406C3.75 8.49219 4.00781 8.75 4.32031 8.75C4.48438 8.75 4.63281 8.69531 4.73438 8.58594L6.375 6.92969L8.02344 8.58594C8.125 8.69531 8.27344 8.75 8.42969 8.75C8.75781 8.75 9.01562 8.49219 9.01562 8.16406C9.01562 8.00781 8.96094 7.86719 8.84375 7.75781L7.19531 6.11719L8.85156 4.46094C8.97656 4.34375 9.02344 4.21094 9.02344 4.05469C9.02344 3.73438 8.76562 3.48438 8.44531 3.48438C8.29688 3.48438 8.17188 3.53906 8.05469 3.64844Z" fill="currentColor" fill-opacity="0.85"/>
     </g>
    </svg>
    """
    }

    public static let failure = Self("Failure") {
    """
    <?xml version="1.0" encoding="UTF-8"?>
    <!--Generator: Apple Native CoreSVG 326-->
    <!DOCTYPE svg
    PUBLIC "-//W3C//DTD SVG 1.1//EN"
           "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
    <svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 0 14.4454 14.047">
     <g>
      <rect height="14.047" opacity="0" width="14.4454" x="0" y="0"/>
      <path d="M8.42192 0.714893L13.3204 5.61333C14.2657 6.55864 14.2813 7.48052 13.3438 8.41021L8.41411 13.3477C7.47661 14.2774 6.55474 14.2696 5.61724 13.3243L0.710986 8.42583C-0.226514 7.48052-0.242139 6.56646 0.695361 5.62896L5.62505 0.691455C6.56255-0.238232 7.48442-0.23042 8.42192 0.714893ZM8.69536 4.55864L7.02349 6.22271L5.35942 4.56646C5.25005 4.46489 5.12505 4.41021 4.9688 4.41021C4.64849 4.41021 4.39067 4.65239 4.39067 4.98052C4.39067 5.12896 4.45317 5.26958 4.56255 5.37896L6.21099 7.02739L4.56255 8.67583C4.45317 8.78521 4.39067 8.92583 4.39067 9.07427C4.39067 9.40239 4.64849 9.66021 4.9688 9.66021C5.12505 9.66021 5.27349 9.60552 5.38286 9.49614L7.02349 7.83989L8.67192 9.49614C8.76567 9.60552 8.92192 9.66021 9.07817 9.66021C9.39849 9.66021 9.6563 9.40239 9.6563 9.07427C9.6563 8.91802 9.60161 8.77739 9.49224 8.66802L7.8438 7.02739L9.50005 5.37114C9.61724 5.25396 9.66411 5.12114 9.66411 4.96489C9.66411 4.64458 9.4063 4.39458 9.0938 4.39458C8.93755 4.39458 8.81255 4.44927 8.69536 4.55864Z" fill="currentColor" fill-opacity="0.85"/>
     </g>
    </svg>
    """
    }

    public static let warning = Self("Warning") {
    """
    <?xml version="1.0" encoding="UTF-8"?>
    <!--Generator: Apple Native CoreSVG 326-->
    <!DOCTYPE svg
    PUBLIC "-//W3C//DTD SVG 1.1//EN"
    "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">
    <svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" viewBox="0 -1 13.2969 12.6953">
    <g transform="translate(0 -1)">
    <rect height="11.6953" opacity="0" width="13.2969" x="0" y="0"/>
    <path d="M7.94531 0.882812L12.6484 9.04688C12.8047 9.32031 12.8906 9.625 12.8906 9.91406C12.8906 10.9062 12.2266 11.6641 11.1406 11.6641L1.75 11.6641C0.664062 11.6641 0 10.9062 0 9.91406C0 9.625 0.078125 9.32812 0.242188 9.04688L4.94531 0.882812C5.27344 0.304688 5.85156 0 6.44531 0C7.03906 0 7.60938 0.304688 7.94531 0.882812ZM5.69531 8.95312C5.69531 9.35156 6.04688 9.66406 6.44531 9.66406C6.84375 9.66406 7.1875 9.35938 7.1875 8.95312C7.1875 8.55469 6.84375 8.23438 6.44531 8.23438C6.03906 8.23438 5.69531 8.55469 5.69531 8.95312ZM5.78906 3.65625L5.875 6.98438C5.88281 7.34375 6.08594 7.54688 6.44531 7.54688C6.78906 7.54688 6.98438 7.35156 6.99219 6.98438L7.09375 3.66406C7.10156 3.29688 6.8125 3.03125 6.4375 3.03125C6.05469 3.03125 5.78125 3.28906 5.78906 3.65625Z" fill="currentColor" fill-opacity="0.85"/>
    </g>
    </svg>
    """
    }
}




infix operator <|

private func <| <A, B> (f: (A) -> B, a: A) -> B {
    return f(a)
}

func base64EncodedString(_ string: String) -> String {
    return Data(string.utf8).base64EncodedString()
}

public let checkmarkSvgBase64 = base64EncodedString(
  #"""
  <svg height="17" viewBox="0 0 18 17" width="18" xmlns="http://www.w3.org/2000/svg"><path d="m2747.82514 923.69487c.73516 0 1.30748-.275483 1.69901-.867077l8.34707-12.788586c.27936-.419937.39488-.839159.39488-1.217309 0-1.042822-.79549-1.821898-1.8564-1.821898-.71796 0-1.17652.260056-1.61611.961902l-7.00525 11.174935-3.51272-4.269522c-.38651-.459578-.8269-.673443-1.43122-.673443-1.06139 0-1.8444.784817-1.8444 1.838933 0 .477534.13316.87095.53417 1.335158l4.65426 5.568605c.4429.52473.96574.758302 1.63671.758302z" fill="#fff" fill-rule="evenodd" transform="translate(-2741 -907)"/></svg>
  """#)

public let collectionIconSvgBase64 = base64EncodedString(
  #"""
  <svg width="20" height="20" xmlns="http://www.w3.org/2000/svg"><path d="M10 .017L19.983 10 10 19.983.017 10 10 .017zM8.336 11.664c.876.875 2.39.938 3.328 0 .95-.951.875-2.452 0-3.328-.876-.875-2.377-.95-3.328 0-.938.938-.875 2.452 0 3.328z" fill="#121212" fill-rule="evenodd"/></svg>
  """#)

public let circleLockSvgBase64 = base64EncodedString(
  """
  <svg height="48" viewBox="0 0 48 48" width="48" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><circle cx="24" cy="24" fill="#121212" r="24"/><path d="m12 17c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm6-9h-1v-2c0-2.76-2.24-5-5-5s-5 2.24-5 5h1.9c0-1.71 1.39-3.1 3.1-3.1s3.1 1.39 3.1 3.1v2h-9.1c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2v-10c0-1.1-.9-2-2-2zm0 12h-12v-10h12z" fill="#fff" transform="translate(12 12)"/></g></svg>
  """)

public let lockSvgBase64 = base64EncodedString(
  """
  <svg height="15" viewBox="0 0 12 15" width="12" xmlns="http://www.w3.org/2000/svg"><path d="m12 5.33333333h-.6666667v-1.33333333c0-1.84-1.4933333-3.33333333-3.3333333-3.33333333s-3.33333333 1.49333333-3.33333333 3.33333333v1.33333333h-.66666667c-.73333333 0-1.33333333.6-1.33333333 1.33333334v6.66666663c0 .7333334.6 1.3333334 1.33333333 1.3333334h8c.7333333 0 1.3333333-.6 1.3333333-1.3333334v-6.66666663c0-.73333334-.6-1.33333334-1.3333333-1.33333334zm-4 5.99999997c-.73333333 0-1.33333333-.6-1.33333333-1.3333333 0-.73333333.6-1.33333333 1.33333333-1.33333333s1.33333333.6 1.33333333 1.33333333c0 .7333333-.6 1.3333333-1.33333333 1.3333333zm2.0666667-5.99999997h-4.13333337v-1.33333333c0-1.14.92666667-2.06666667 2.06666667-2.06666667s2.0666667.92666667 2.0666667 2.06666667z" fill="#7d7d7d" fill-rule="evenodd" transform="translate(-2)"/></svg>
  """)

public let unlockSvgBase64 = base64EncodedString(
  """
  <svg height="15" viewBox="0 0 12 15" width="12" xmlns="http://www.w3.org/2000/svg"><path d="m8 11.3333333c.73333333 0 1.33333333-.6 1.33333333-1.3333333 0-.73333333-.6-1.33333333-1.33333333-1.33333333s-1.33333333.6-1.33333333 1.33333333c0 .7333333.6 1.3333333 1.33333333 1.3333333zm4-5.99999997h-.6666667v-1.33333333c0-1.84-1.4933333-3.33333333-3.3333333-3.33333333s-3.33333333 1.49333333-3.33333333 3.33333333h1.26666666c0-1.14.92666667-2.06666667 2.06666667-2.06666667s2.0666667.92666667 2.0666667 2.06666667v1.33333333h-6.0666667c-.73333333 0-1.33333333.6-1.33333333 1.33333334v6.66666663c0 .7333334.6 1.3333334 1.33333333 1.3333334h8c.7333333 0 1.3333333-.6 1.3333333-1.3333334v-6.66666663c0-.73333334-.6-1.33333334-1.3333333-1.33333334zm0 7.99999997h-8v-6.66666663h8z" fill="#7d7d7d" fill-rule="evenodd" transform="translate(-2)"/></svg>
  """)

public func pointFreeTextDiamondLogoSvgBase64(fill: String = "#fff") -> String {
    base64EncodedString(
    """
    <svg height="16" viewBox="0 0 142 16" width="142" xmlns="http://www.w3.org/2000/svg"><path d="m43.3084787.05876394.0700608.05142528 9.7604979 7.89433005v-7.64763224h3.9087708v15.47468107c0 .1614749-.0555457.2321202-.2499556.0974905l-.0700607-.0526364-9.7604979-7.87190293v7.60277813h-3.9087709v-15.45225403c0-.1211062.0555457-.22404647.2499556-.09627943zm54.1243498.51266463v3.43134232h-6.4003265v3.5434777h5.6460023v3.43134231h-5.6460023v4.844248h-4.1602123v-15.25041033zm9.1792435 0c4.183071 0 6.537477 2.60154059 6.537477 5.98802876 0 2.19785325-.982908 3.94716507-2.811572 4.93395627l2.765855 4.3284253h-4.503087l-2.171539-3.4986236h-1.554365v3.4986236h-4.160212v-15.25041033zm20.608399 0v3.43134232h-6.743202v3.5434777h5.623144v3.25192571h-5.623144v1.5923223h7.063218v3.4313423h-11.22343v-15.25041033zm14.459513 0v3.43134232h-6.743201v3.5434777h5.623144v3.25192571h-5.623144v1.5923223h7.063217v3.4313423h-11.22343v-15.25041033zm-118.3971636-.39395813c4.4802286 0 8.3432828 3.20707158 8.3432828 7.80462176 0 4.5975502-3.8630542 7.8270488-8.3432828 7.8270488-4.4802285 0-8.3432828-3.2294986-8.3432828-7.8270488 0-4.59755018 3.8630543-7.80462176 8.3432828-7.80462176zm-17.38537669.17941659c4.18307059 0 6.53747639 2.60154059 6.53747639 5.98802876 0 3.38648818-2.3544058 5.96560171-6.53747639 5.96560171h-1.73723148v3.2967799h-4.16021223v-15.25041037zm33.14781349 0v15.25041037h-4.1602123v-15.25041037zm33.4906881 0v3.43134233h-4.1144956v11.81906804h-4.1602123v-11.81906804h-4.1144956v-3.43134233zm5.6956595 5.54249465 3.2544627 3.25446279-3.2324881 3.23248813-3.2544628-3.25446278zm-54.9487844-2.17843355c-2.6058472 0-4.0916373 1.86144715-4.0916373 4.26114407 0 2.3996969 1.4857901 4.2835711 4.0916373 4.2835711s4.0916373-1.8838742 4.0916373-4.2835711c0-2.39969692-1.4857901-4.26114407-4.0916373-4.26114407zm83.1235266.30424984h-1.531506v5.04609165h1.531506c1.874382 0 2.560131-1.0316454 2.560131-2.51183229 0-1.48018688-.685749-2.53425936-2.560131-2.53425936zm-100.71462807-.21454154h-1.5315067v5.04609165h1.5315067c1.87438134 0 2.5601306-1.0316454 2.5601306-2.51183229 0-1.48018688-.68574926-2.53425936-2.5601306-2.53425936z" fill="\(fill)" fill-rule="evenodd"/></svg>
    """)
}

public func playIconSvgBase64(fill: String = "121212") -> String {
    base64EncodedString(
    """
    <svg width="20" height="20" xmlns="http://www.w3.org/2000/svg">
      <g fill="none" fill-rule="evenodd">
        <path d="M8 14.5l6-4.5-6-4.5v9zM10 0C4.48 0 0 4.48 0 10s4.48 10 10 10 10-4.48 10-10S15.52 0 10 0zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8z" fill="#\(fill)"/>
      </g>
    </svg>
    """)
}

public let exercisesIconSvgBase64 = base64EncodedString(
  """
  <svg width="16" height="18" xmlns="http://www.w3.org/2000/svg">
    <g fill="none" fill-rule="evenodd">
      <path fill="#FFF" d="M-75-1329H949v3583H-75z"/>
      <path d="M13.8333 2.5H10.35C10 1.5333 9.0833.8333 8 .8333c-1.0833 0-2 .7-2.35 1.6667H2.1667C1.25 2.5.5 3.25.5 4.1667v11.6666C.5 16.75 1.25 17.5 2.1667 17.5h11.6666c.9167 0 1.6667-.75 1.6667-1.6667V4.1667C15.5 3.25 14.75 2.5 13.8333 2.5zM8 2.5c.4583 0 .8333.375.8333.8333 0 .4584-.375.8334-.8333.8334-.4583 0-.8333-.375-.8333-.8334 0-.4583.375-.8333.8333-.8333zm1.6667 11.6667H3.8333V12.5h5.8334v1.6667zm2.5-3.3334H3.8333V9.1667h8.3334v1.6666zm0-3.3333H3.8333V5.8333h8.3334V7.5z" fill="#000"/>
    </g>
  </svg>
  """)

public func hourGlassSvgBase64(fill: String = "000") -> String {
    base64EncodedString(
    """
    <svg height="20" viewBox="0 0 45 68" width="20" xmlns="http://www.w3.org/2000/svg"><path fill="#\(fill)" d="m5.3555 67.4375h34.7656c2.9785 0 4.541-1.61133 4.5898-4.58984l.0489-2.2461c.1953-9.03316-11.084-19.28706-16.1621-23.33986-1.4161-1.123-1.9043-2.1972-1.9043-3.4668 0-1.2695.4394-2.1972 1.9043-3.4668 5.0293-4.3945 16.1132-14.1601 16.1132-23.2422v-2.2461c0-2.9785-1.6113-4.5898-4.5898-4.5898h-34.7656c-2.9785 0-4.58988 1.6113-4.58988 4.5898v2.2461c0 9.0821 11.08398 18.8965 16.16208 23.2422 1.4649 1.2696 1.9043 2.1973 1.9043 3.4668 0 1.2696-.4882 2.3438-1.9043 3.4668-5.0781 4.0528-16.16208 14.3067-16.16208 23.33986v2.2461c0 2.97851 1.61138 4.58984 4.58988 4.58984zm3.0273-3.07617c-1.3672 0-1.9043-1.26953-.6836-2.2461l12.793-10.54683c.3906-.3418.7324-.586.7324-1.1231v-17.9199c0-1.9531-.5859-2.8809-1.8066-3.9551-1.8067-1.709-6.0059-5.1269-7.8125-7.7148-.586-.7813-.4883-1.4649.2441-1.4649h21.7285c.7324 0 .8301.6836.2442 1.4649-1.8067 2.5879-5.9571 6.0547-7.8125 7.6172-1.3184 1.0742-1.8067 2.0996-1.8067 4.0527v17.9199c0 .5371.3907.7813.7813 1.1231l12.7929 10.54683c1.2696.97657.6836 2.2461-.6835 2.2461z" transform="translate(-.052)"/></svg>
    """)
}

public let referencesIconSvgBase64 = base64EncodedString(
  """
  <svg width="16" height="10" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path fill="#FFF" d="M-75-1370H949v3583H-75z"/><path d="M.5 5.8333h1.6667V4.1667H.5v1.6666zm0 3.3334h1.6667V7.5H.5v1.6667zM.5 2.5h1.6667V.8333H.5V2.5zm3.3333 3.3333H15.5V4.1667H3.8333v1.6666zm0 3.3334H15.5V7.5H3.8333v1.6667zm0-8.3334V2.5H15.5V.8333H3.8333z" fill="#000"/></g></svg>
  """)

public let downloadIconSvgBase64 = base64EncodedString(
  """
  <svg width="12" height="15" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path fill="#FFF" d="M-77-1403H947v3583H-77z"/><path d="M11.8333 5.5H8.5v-5h-5v5H.1667L6 11.3333 11.8333 5.5zM.1667 13v1.6667h11.6666V13H.1667z" fill="#000"/></g></svg>
  """)

public let questionIconSvgBase64 = base64EncodedString(
  """
  <svg height="16.2981" width="16.2977" xmlns="http://www.w3.org/2000/svg"><path d="m0 0h16.2977v16.2981h-16.2977z" opacity="0"/><path d="m8.14762 16.2952c4.46868 0 8.15008-3.6842 8.15008-8.14758 0-4.46868-3.6867-8.14762-8.15541-8.14762-4.46087 0-8.14229 3.67894-8.14229 8.14762 0 4.46338 3.68675 8.14758 8.14762 8.14758zm0-1.7902c-3.52602 0-6.35208-2.8314-6.35208-6.35738 0-3.52602 2.82073-6.35741 6.34675-6.35741 3.52601 0 6.36521 2.83139 6.36521 6.35741 0 3.52598-2.8339 6.35738-6.35988 6.35738zm-.18642-4.84453c.45273 0 .75951-.25497.79429-.57953.00249-.02877.00497-.06783.00994-.09127.03727-.38036.32453-.6467.80681-.95888.78516-.50497 1.24966-.94815 1.24966-1.83845 0-1.29936-1.18108-2.07702-2.61354-2.07702-1.3363 0-2.26135.59591-2.52197 1.33702-.05149.13742-.07989.27484-.07989.42256 0 .39628.30573.66328.6654.66328.29792 0 .51203-.12003.67502-.33486l.10832-.14526c.25854-.40096.60019-.60448 1.03171-.60448.5743 0 .96568.3516.96568.83107 0 .47163-.32564.69963-.97055 1.14638-.55256.38565-.94959.77736-.94959 1.44361v.07424c0 .46835.2986.71159.82871.71159zm-.00817 2.43993c.53759 0 .96548-.3693.96548-.892 0-.5202-.42256-.892-.96548-.892s-.97294.3746-.97294.892c0 .5173.43535.892.97294.892z" fill-opacity=".85"/></svg>
  """)

public let leftNavigationChevronSvgBase64 =
base64EncodedString <| """
    <svg height="8" viewBox="0 0 6 8" width="6" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path d="m-57-78h1024v1649h-1024z" fill="#fff"/><path d="m-57-22h1024v412h-1024z" fill="#121212"/><path d="m10.2733333 4.94-.93999997-.94-4 4 4 4 .93999997-.94-3.0533333-3.06z" fill="#a8a8a8" transform="translate(-5 -4)"/></g></svg>
    """

public let leftChevronSvgBase64 =
base64EncodedString <| """
    <svg height="12" viewBox="0 0 8 12" width="8" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd"><path d="m-215-1599h1024v1649h-1024z" fill="#fff"/><path d="m0 0h1024v381h-1024z" fill="#fafafa" transform="matrix(1 0 0 -1 -215 50)"/><path d="m15.41 7.41-1.41-1.41-6 6 6 6 1.41-1.41-4.58-4.59z" fill="#000" transform="translate(-8 -6)"/></g></svg>
    """

public let rightChevronSvgBase64 =
base64EncodedString <| """
    <svg height="12" width="8" xmlns="http://www.w3.org/2000/svg"><path d="m.59 1.41 4.58 4.59-4.58 4.59 1.41 1.41 6-6-6-6z" fill-rule="evenodd"/></svg>
    """

public let collectionsIconSvgBase64 = base64EncodedString(
  #"""
  <svg width="132" height="80" xmlns="http://www.w3.org/2000/svg"><g fill="#FFF" fill-rule="evenodd"><path d="M40 .07L79.93 40 40 79.93.07 40 40 .07z" opacity=".25"/><path d="M94.93 29H82v22h12.93L66 79.93 26.07 40 66 .07 94.93 29zm7.07 7.07l3.93 3.93-3.93 3.93v-7.86z" opacity=".5"/><path d="M92 .07L131.93 40 92 79.93 52.07 40 92 .07zm-6.655 46.585c3.503 3.503 9.557 3.753 13.31 0 3.803-3.803 3.503-9.807 0-13.31s-9.507-3.803-13.31 0c-3.753 3.753-3.503 9.807 0 13.31z"/></g></svg>
  """#)

public func pointFreeTextLogoSvgBase64(color: String) -> String {
    return base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="238px" height="28px" viewBox="0 0 238 28" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Group-2" fill="\(color)">
                <path d="M7.23121318,21.5434056 L7.23121318,27.3127704 L0,27.3127704 L0,0.624552309 L10.2508407,0.624552309 C17.5217858,0.624552309 21.6141757,5.17724833 21.6141757,11.1036026 C21.6141757,17.0299569 17.5217858,21.5434056 10.2508407,21.5434056 L7.23121318,21.5434056 Z M7.23121318,15.4993091 L9.8932532,15.4993091 C13.1512723,15.4993091 14.3432305,13.6939297 14.3432305,11.1036026 C14.3432305,8.51327559 13.1512723,6.66864875 9.8932532,6.66864875 L7.23121318,6.66864875 L7.23121318,15.4993091 Z M40.4698195,27.6659968 C32.6823591,27.6659968 25.9676612,22.0143741 25.9676612,13.9686613 C25.9676612,5.92294854 32.6823591,0.310573273 40.4698195,0.310573273 C48.2572798,0.310573273 54.9719778,5.92294854 54.9719778,13.9686613 C54.9719778,22.0143741 48.2572798,27.6659968 40.4698195,27.6659968 Z M40.4698195,21.4649108 C44.9992607,21.4649108 47.5818368,18.1681309 47.5818368,13.9686613 C47.5818368,9.76919173 44.9992607,6.51165924 40.4698195,6.51165924 C35.9403782,6.51165924 33.3578021,9.76919173 33.3578021,13.9686613 C33.3578021,18.1681309 35.9403782,21.4649108 40.4698195,21.4649108 Z M60.6366173,27.3127704 L67.8678305,27.3127704 L67.8678305,0.624552309 L60.6366173,0.624552309 L60.6366173,27.3127704 Z M98.6033245,27.7837389 C99.0403759,28.1369653 99.1595717,28.0192232 99.1595717,27.7052442 L99.1595717,0.624552309 L92.3654098,0.624552309 L92.3654098,14.0079087 L75.3998712,0.192831135 C74.9628199,-0.160395281 74.843624,0.0358416168 74.843624,0.271325894 L74.843624,27.3127704 L81.6377859,27.3127704 L81.6377859,14.0079087 L98.6033245,27.7837389 Z M111.697837,27.3127704 L118.92905,27.3127704 L118.92905,6.62940137 L126.080799,6.62940137 L126.080799,0.624552309 L104.546088,0.624552309 L104.546088,6.62940137 L111.697837,6.62940137 L111.697837,27.3127704 Z M131.427583,20.758458 C134.208819,20.758458 136.711932,18.48211 136.711932,15.5385565 C136.711932,12.5557557 134.208819,10.318655 131.427583,10.318655 C128.646348,10.318655 126.143235,12.5557557 126.143235,15.5385565 C126.143235,18.48211 128.646348,20.758458 131.427583,20.758458 Z M142.177911,27.3127704 L149.409125,27.3127704 L149.409125,18.8353364 L159.222914,18.8353364 L159.222914,12.8304873 L149.409125,12.8304873 L149.409125,6.62940137 L160.534068,6.62940137 L160.534068,0.624552309 L142.177911,0.624552309 L142.177911,27.3127704 Z M182.965586,19.7380261 L187.773151,27.3127704 L179.945959,27.3127704 L176.171424,21.1901792 L173.469653,21.1901792 L173.469653,27.3127704 L166.238439,27.3127704 L166.238439,0.624552309 L176.48928,0.624552309 C183.760225,0.624552309 187.852615,5.17724833 187.852615,11.1036026 C187.852615,14.9498458 186.144142,18.0111414 182.965586,19.7380261 Z M173.469653,15.4993091 L176.131693,15.4993091 C179.389712,15.4993091 180.58167,13.6939297 180.58167,11.1036026 C180.58167,8.51327559 179.389712,6.66864875 176.131693,6.66864875 L173.469653,6.66864875 L173.469653,15.4993091 Z M193.358327,27.3127704 L212.86671,27.3127704 L212.86671,21.3079213 L200.58954,21.3079213 L200.58954,18.5213574 L210.363597,18.5213574 L210.363597,12.8304873 L200.58954,12.8304873 L200.58954,6.62940137 L212.310462,6.62940137 L212.310462,0.624552309 L193.358327,0.624552309 L193.358327,27.3127704 Z M218.491617,27.3127704 L238,27.3127704 L238,21.3079213 L225.72283,21.3079213 L225.72283,18.5213574 L235.496888,18.5213574 L235.496888,12.8304873 L225.72283,12.8304873 L225.72283,6.62940137 L237.443753,6.62940137 L237.443753,0.624552309 L218.491617,0.624552309 L218.491617,27.3127704 Z" id="logo"></path>
            </g>
        </g>
    </svg>
    """
}

public func pointFreeDiamondLogoSvgBase64(fill: String) -> String {
    return base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="72px" height="72px" viewBox="0 0 72 72" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Group" fill="\(fill)">
                <path d="M54.3363286,53.6636714 L36,72 L17.6636714,53.6636714 L18.1191186,53.6636714 L18.1191186,49.6912661 L22.637493,49.6912661 L22.637493,46.8774789 L18.1191186,46.8774789 L18.1191186,43.971738 L23.2411625,43.971738 L23.2411625,41.1579509 L14.7897901,41.1579509 L14.7897901,50.7897901 L2.13162821e-14,36 L7.79728559,28.2027144 L7.79728559,35.2729059 L11.1266141,35.2729059 L11.1266141,32.5694634 L12.5168831,32.5694634 C15.8645047,32.5694634 17.7486851,30.4545254 17.7486851,27.6775198 C17.7486851,25.0959129 16.1203297,23.0705875 13.2016474,22.7983526 L36,2.13162821e-14 L58.7671854,22.7671854 L54.2981267,22.7671854 L54.2981267,25.5809725 L57.5908691,25.5809725 L57.5908691,35.2729059 L60.9201976,35.2729059 L60.9201976,25.5809725 L61.5809725,25.5809725 L72,36 L57.1501157,50.8498843 L52.0297929,50.8498843 L52.0297929,49.5441399 L56.5298743,49.5441399 L56.5298743,46.8774789 L52.0297929,46.8774789 L52.0297929,43.971738 L57.426232,43.971738 L57.426232,41.1579509 L48.7004644,41.1579509 L48.7004644,53.6636714 L54.3363286,53.6636714 Z M11.1266141,29.7372855 L11.1266141,25.5993633 L12.352246,25.5993633 C13.8522732,25.5993633 14.4010636,26.4637293 14.4010636,27.6775198 C14.4010636,28.8913103 13.8522732,29.7372855 12.352246,29.7372855 L11.1266141,29.7372855 Z M26.0217006,35.4384228 C29.6071313,35.4384228 32.6986506,32.7901526 32.6986506,29.0200457 C32.6986506,25.2499387 29.6071313,22.6200593 26.0217006,22.6200593 C22.4362699,22.6200593 19.3447505,25.2499387 19.3447505,29.0200457 C19.3447505,32.7901526 22.4362699,35.4384228 26.0217006,35.4384228 Z M26.0217006,32.5326819 C23.936297,32.5326819 22.7472511,30.9878576 22.7472511,29.0200457 C22.7472511,27.0522338 23.936297,25.5258002 26.0217006,25.5258002 C28.1071042,25.5258002 29.29615,27.0522338 29.29615,29.0200457 C29.29615,30.9878576 28.1071042,32.5326819 26.0217006,32.5326819 Z M34.8983855,35.2729059 L38.227714,35.2729059 L38.227714,22.7671854 L34.8983855,22.7671854 L34.8983855,35.2729059 Z M51.9703406,35.4935951 C52.1715638,35.659112 52.2264428,35.6039397 52.2264428,35.4568136 L52.2264428,22.7671854 L49.0983375,22.7671854 L49.0983375,29.0384364 L41.2872206,22.564887 C41.0859974,22.3993701 41.0311184,22.4913239 41.0311184,22.6016685 L41.0311184,35.2729059 L44.1592238,35.2729059 L44.1592238,29.0384364 L51.9703406,35.4935951 Z M33.1605492,50.1142537 C34.6239903,49.30506 35.4105899,47.8705803 35.4105899,46.0682853 C35.4105899,43.2912797 33.5264095,41.1579509 30.178788,41.1579509 L25.4591904,41.1579509 L25.4591904,53.6636714 L28.7885189,53.6636714 L28.7885189,50.794712 L30.0324439,50.794712 L31.7702802,53.6636714 L35.3740039,53.6636714 L33.1605492,50.1142537 Z M28.7885189,48.128051 L28.7885189,43.9901288 L30.0141508,43.9901288 C31.514178,43.9901288 32.0629684,44.8544947 32.0629684,46.0682853 C32.0629684,47.2820758 31.514178,48.128051 30.0141508,48.128051 L28.7885189,48.128051 Z M37.5371528,53.6636714 L46.5190225,53.6636714 L46.5190225,50.8498843 L40.8664813,50.8498843 L40.8664813,49.5441399 L45.3665627,49.5441399 L45.3665627,46.8774789 L40.8664813,46.8774789 L40.8664813,43.971738 L46.2629203,43.971738 L46.2629203,41.1579509 L37.5371528,41.1579509 L37.5371528,53.6636714 Z" id="glyph"></path>
            </g>
        </g>
    </svg>
    """
}

public let thingiesSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="124px" height="112px" viewBox="0 0 124 112" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <path d="M123.974358,104.706007 L123.974358,104.757291 C123.974358,108.773092 120.718908,112.028541 116.703107,112.028541 C112.687307,112.028541 109.431857,108.773092 109.431857,104.757291 L109.431857,104.706007 C109.431857,100.690206 112.687307,97.4347562 116.703107,97.4347562 C120.718908,97.4347562 123.974358,100.690206 123.974358,104.706007 Z M53.9241897,7.27125046 L53.9241897,7.32253459 C53.9241897,11.3383353 50.6687399,14.5937851 46.6529392,14.5937851 C42.6371385,14.5937851 39.3816887,11.3383353 39.3816887,7.32253459 L39.3816887,7.27125046 C39.3816887,3.25544972 42.6371385,1.09440699e-13 46.6529392,1.09686595e-13 C50.6687399,1.09932492e-13 53.9241897,3.25544972 53.9241897,7.27125046 Z M-13.9292182,65.4668962 L118.910125,65.4668962 C121.721185,65.4668962 124,67.745711 124,70.5567716 C124,73.3678321 121.721185,75.6466469 118.910125,75.6466469 L-13.9292182,75.6466469 C-16.7402787,75.6466469 -19.0190935,73.3678321 -19.0190935,70.5567716 C-19.0190935,67.745711 -16.7402787,65.4668962 -13.9292182,65.4668962 Z M-71.9101247,100.368898 L60.9292182,100.368898 C63.7402787,100.368898 66.0190935,102.647713 66.0190935,105.458774 C66.0190935,108.269834 63.7402787,110.548649 60.9292182,110.548649 L-71.9101247,110.548649 C-74.7211852,110.548649 -77,108.269834 -77,105.458774 C-77,102.647713 -74.7211852,100.368898 -71.9101247,100.368898 Z M-70.3708924,30.564894 L62.4684505,30.564894 C65.279511,30.564894 67.5583258,32.8437088 67.5583258,35.6547693 C67.5583258,38.4658299 65.279511,40.7446447 62.4684505,40.7446447 L-70.3708924,40.7446447 C-73.1819529,40.7446447 -75.4607677,38.4658299 -75.4607677,35.6547693 C-75.4607677,32.8437088 -73.1819529,30.564894 -70.3708924,30.564894 Z" id="lines" fill="#974DFF"></path>
        </g>
    </svg>
    """

public let heroMountainSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="615px" height="231px" viewBox="0 0 615 231" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <defs>
            <rect id="path-1" x="73.7269344" y="73.7269344" width="355.98513" height="355.98513"></rect>
            <rect id="path-3" x="73.7269344" y="73.7269344" width="355.98513" height="355.98513"></rect>
        </defs>
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="mountains-copy" transform="translate(-99.000000, 0.000000)">
                <rect id="Mask" fill="#FFF080" transform="translate(568.000000, 231.000000) rotate(-315.000000) translate(-568.000000, -231.000000) " x="465" y="128" width="206" height="206"></rect>
                <rect id="rect" fill="#79F2B0" transform="translate(223.500000, 221.500000) rotate(-315.000000) translate(-223.500000, -221.500000) " x="67" y="65" width="313" height="313"></rect>
                <rect id="rect-copy-2" fill="#4CCCFF" transform="translate(346.500000, 186.500000) rotate(-315.000000) translate(-346.500000, -186.500000) " x="235" y="75" width="223" height="223"></rect>
                <rect id="rect-copy" fill="#FFF080" transform="translate(183.000000, 223.000000) rotate(-315.000000) translate(-183.000000, -223.000000) " x="54" y="94" width="258" height="258"></rect>
                <g id="bg-square-2" transform="translate(203.000000, 61.000000)">
                    <mask id="mask-2" fill="white">
                        <use xlink:href="#path-1"></use>
                    </mask>
                    <use id="Mask" fill="#79F2B0" transform="translate(251.719500, 251.719500) rotate(-315.000000) translate(-251.719500, -251.719500) " xlink:href="#path-1"></use>
                </g>
                <g id="Group" transform="translate(48.000000, 76.000000)">
                    <mask id="mask-4" fill="white">
                        <use xlink:href="#path-3"></use>
                    </mask>
                    <use id="Mask" fill="#974DFF" transform="translate(251.719500, 251.719500) rotate(-315.000000) translate(-251.719500, -251.719500) " xlink:href="#path-3"></use>
                    <rect id="Rectangle-6-Copy" fill="#FFFFFF" mask="url(#mask-4)" transform="translate(322.049123, 194.628362) rotate(-315.000000) translate(-322.049123, -194.628362) " x="233.525766" y="106.105005" width="177.046714" height="177.046714"></rect>
                </g>
            </g>
        </g>
    </svg>
    """

public let pointFreeHeroSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="424px" height="112px" viewBox="0 0 424 112" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="pf-logo-hero">
                <path d="M200.988627,104.717114 L200.988627,104.73986 C200.988627,108.756086 197.732832,112.011881 193.716606,112.011881 C189.700379,112.011881 186.444584,108.756086 186.444584,104.73986 L186.444584,104.717114 C186.444584,100.700888 189.700379,97.4450926 193.716606,97.4450926 C197.732832,97.4450926 200.988627,100.700888 200.988627,104.717114 Z M131.068041,7.27202183 L131.068041,7.29476709 C131.068041,11.3109939 127.812245,14.5667889 123.796019,14.5667889 C119.779792,14.5667889 116.523997,11.3109939 116.523997,7.29476709 L116.523997,7.27202183 C116.523997,3.25579507 119.779792,5.79261824e-14 123.796019,5.81721054e-14 C127.812245,5.84180283e-14 131.068041,3.25579507 131.068041,7.27202183 Z M63.3358838,65.4595691 L195.909585,65.4595691 C198.720943,65.4595691 201,67.7386257 201,70.5499844 C201,73.3613431 198.720943,75.6403997 195.909585,75.6403997 L63.3358838,75.6403997 C60.5245251,75.6403997 58.2454685,73.3613431 58.2454685,70.5499844 C58.2454685,67.7386257 60.5245251,65.4595691 63.3358838,65.4595691 Z M5.46223258,100.365274 L138.035934,100.365274 C140.847292,100.365274 143.126349,102.64433 143.126349,105.455689 C143.126349,108.267048 140.847292,110.546104 138.035934,110.546104 L5.46223258,110.546104 C2.65087385,110.546104 0.371817298,108.267048 0.371817298,105.455689 C0.371817298,102.64433 2.65087385,100.365274 5.46223258,100.365274 Z M6.99861754,30.5538643 L139.572318,30.5538643 C142.383677,30.5538643 144.662734,32.8329209 144.662734,35.6442796 C144.662734,38.4556383 142.383677,40.7346949 139.572318,40.7346949 L6.99861754,40.7346949 C4.18725881,40.7346949 1.90820226,38.4556383 1.90820226,35.6442796 C1.90820226,32.8329209 4.18725881,30.5538643 6.99861754,30.5538643 Z" id="lines" fill="#FFFFFF"></path>
                <g id="Group-2" transform="translate(115.000000, 55.000000)" fill="#121212">
                    <path d="M7.23121318,21.5434056 L7.23121318,27.3127704 L0,27.3127704 L0,0.624552309 L10.2508407,0.624552309 C17.5217858,0.624552309 21.6141757,5.17724833 21.6141757,11.1036026 C21.6141757,17.0299569 17.5217858,21.5434056 10.2508407,21.5434056 L7.23121318,21.5434056 Z M7.23121318,15.4993091 L9.8932532,15.4993091 C13.1512723,15.4993091 14.3432305,13.6939297 14.3432305,11.1036026 C14.3432305,8.51327559 13.1512723,6.66864875 9.8932532,6.66864875 L7.23121318,6.66864875 L7.23121318,15.4993091 Z M40.4698195,27.6659968 C32.6823591,27.6659968 25.9676612,22.0143741 25.9676612,13.9686613 C25.9676612,5.92294854 32.6823591,0.310573273 40.4698195,0.310573273 C48.2572798,0.310573273 54.9719778,5.92294854 54.9719778,13.9686613 C54.9719778,22.0143741 48.2572798,27.6659968 40.4698195,27.6659968 Z M40.4698195,21.4649108 C44.9992607,21.4649108 47.5818368,18.1681309 47.5818368,13.9686613 C47.5818368,9.76919173 44.9992607,6.51165924 40.4698195,6.51165924 C35.9403782,6.51165924 33.3578021,9.76919173 33.3578021,13.9686613 C33.3578021,18.1681309 35.9403782,21.4649108 40.4698195,21.4649108 Z M60.6366173,27.3127704 L67.8678305,27.3127704 L67.8678305,0.624552309 L60.6366173,0.624552309 L60.6366173,27.3127704 Z M98.6033245,27.7837389 C99.0403759,28.1369653 99.1595717,28.0192232 99.1595717,27.7052442 L99.1595717,0.624552309 L92.3654098,0.624552309 L92.3654098,14.0079087 L75.3998712,0.192831135 C74.9628199,-0.160395281 74.843624,0.0358416168 74.843624,0.271325894 L74.843624,27.3127704 L81.6377859,27.3127704 L81.6377859,14.0079087 L98.6033245,27.7837389 Z M111.697837,27.3127704 L118.92905,27.3127704 L118.92905,6.62940137 L126.080799,6.62940137 L126.080799,0.624552309 L104.546088,0.624552309 L104.546088,6.62940137 L111.697837,6.62940137 L111.697837,27.3127704 Z M131.427583,20.758458 C134.208819,20.758458 136.711932,18.48211 136.711932,15.5385565 C136.711932,12.5557557 134.208819,10.318655 131.427583,10.318655 C128.646348,10.318655 126.143235,12.5557557 126.143235,15.5385565 C126.143235,18.48211 128.646348,20.758458 131.427583,20.758458 Z M142.177911,27.3127704 L149.409125,27.3127704 L149.409125,18.8353364 L159.222914,18.8353364 L159.222914,12.8304873 L149.409125,12.8304873 L149.409125,6.62940137 L160.534068,6.62940137 L160.534068,0.624552309 L142.177911,0.624552309 L142.177911,27.3127704 Z M182.965586,19.7380261 L187.773151,27.3127704 L179.945959,27.3127704 L176.171424,21.1901792 L173.469653,21.1901792 L173.469653,27.3127704 L166.238439,27.3127704 L166.238439,0.624552309 L176.48928,0.624552309 C183.760225,0.624552309 187.852615,5.17724833 187.852615,11.1036026 C187.852615,14.9498458 186.144142,18.0111414 182.965586,19.7380261 Z M173.469653,15.4993091 L176.131693,15.4993091 C179.389712,15.4993091 180.58167,13.6939297 180.58167,11.1036026 C180.58167,8.51327559 179.389712,6.66864875 176.131693,6.66864875 L173.469653,6.66864875 L173.469653,15.4993091 Z M193.358327,27.3127704 L212.86671,27.3127704 L212.86671,21.3079213 L200.58954,21.3079213 L200.58954,18.5213574 L210.363597,18.5213574 L210.363597,12.8304873 L200.58954,12.8304873 L200.58954,6.62940137 L212.310462,6.62940137 L212.310462,0.624552309 L193.358327,0.624552309 L193.358327,27.3127704 Z M218.491617,27.3127704 L238,27.3127704 L238,21.3079213 L225.72283,21.3079213 L225.72283,18.5213574 L235.496888,18.5213574 L235.496888,12.8304873 L225.72283,12.8304873 L225.72283,6.62940137 L237.443753,6.62940137 L237.443753,0.624552309 L218.491617,0.624552309 L218.491617,27.3127704 Z" id="logo"></path>
                </g>
                <rect id="bounds" fill-opacity="0" fill="#FFFFFF" x="0" y="0" width="495" height="113"></rect>
            </g>
        </g>
    </svg>
    """

public func rightArrowSvgBase64(fill: String) -> String {
    return base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="20px" height="20px" viewBox="0 0 20 20" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <g id="Page-1" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
    <polygon id="right" fill="\(fill)" points="9.57421875 0 19.3828125 9.80859375 9.57421875 19.6171875 8.23828125 18.2929688 15.7851562 10.7460938 0 10.7460938 0 8.87109375 15.7851562 8.87109375 8.23828125 1.32421875"></polygon>
    </g>
    </svg>
    """
}

public let logoSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="371px" height="269px" viewBox="0 0 371 269" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <defs>
            <rect id="path-1" x="28.9705627" y="96.0294373" width="24" height="24"></rect>
        </defs>
        <g id="Episodes" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Sign-up" transform="translate(-112.000000, -112.000000)">
                <g id="logo" transform="translate(112.000000, 112.000000)">
                    <circle id="Oval" stroke="#79F2B0" stroke-width="4" fill-opacity="0" fill="#FFFFFF" cx="250" cy="64" r="36"></circle>
                    <circle id="Oval-Copy-2" stroke="#79F2B0" stroke-width="4" fill-opacity="0" fill="#FFFFFF" cx="104" cy="214" r="36"></circle>
                    <circle id="Oval-Copy" fill="#79F2B0" cx="145" cy="258" r="8"></circle>
                    <rect id="Rectangle-7" fill="#974CFF" transform="translate(18.313708, 88.313708) rotate(-315.000000) translate(-18.313708, -88.313708) " x="10.3137085" y="80.3137085" width="16" height="16"></rect>
                    <rect id="Rectangle-7-Copy" fill="#FFF080" transform="translate(366.000000, 229.000000) rotate(-270.000000) translate(-366.000000, -229.000000) " x="361" y="224" width="10" height="10" rx="5"></rect>
                    <rect id="Rectangle-7-Copy-8" fill="#79F2B0" transform="translate(86.000000, 74.000000) rotate(-270.000000) translate(-86.000000, -74.000000) " x="81" y="69" width="10" height="10" rx="5"></rect>
                    <rect id="Rectangle-7-Copy-6" fill="#FFF080" transform="translate(318.000000, 162.000000) rotate(-270.000000) translate(-318.000000, -162.000000) " x="313" y="157" width="10" height="10" rx="5"></rect>
                    <rect id="Rectangle-7-Copy-4" fill="#974DFF" transform="translate(351.627417, 133.627417) rotate(-315.000000) translate(-351.627417, -133.627417) " x="335.627417" y="117.627417" width="32" height="32" rx="16"></rect>
                    <rect id="Rectangle-7-Copy-2" fill="#4CCCFF" transform="translate(324.970563, 108.970563) rotate(-315.000000) translate(-324.970563, -108.970563) " x="312.970563" y="96.9705627" width="24" height="24" rx="12"></rect>
                    <rect id="Rectangle-7-Copy-7" fill="#4CCCFF" transform="translate(16.970563, 133.769553) rotate(-315.000000) translate(-16.970563, -133.769553) " x="4.97056275" y="121.769553" width="24" height="24"></rect>
                    <rect id="Rectangle-7-Copy-3" fill="#79F2B0" transform="translate(304.000000, 88.000000) rotate(-270.000000) translate(-304.000000, -88.000000) " x="296" y="80" width="16" height="16" rx="8"></rect>
                    <g id="Rectangle-7-Copy-5" transform="translate(40.970563, 108.029437) rotate(-315.000000) translate(-40.970563, -108.029437) ">
                        <use fill-opacity="0" fill="#FFFFFF" fill-rule="evenodd" xlink:href="#path-1"></use>
                        <rect stroke="#79F2B0" stroke-width="4" x="30.9705627" y="98.0294373" width="20" height="20"></rect>
                    </g>
                    <rect id="Rectangle-6" fill="#121212" transform="translate(184.675144, 134.350288) rotate(-315.000000) translate(-184.675144, -134.350288) " x="89.6751442" y="39.3502884" width="190" height="190"></rect>
                    <rect id="Rectangle-8" fill="#FFF080" x="273" y="202" width="98" height="7" rx="3.5"></rect>
                    <rect id="Rectangle-8-Copy-2" fill="#FFF080" x="247" y="226" width="98" height="7" rx="3.5"></rect>
                    <rect id="Rectangle-8-Copy" fill="#FFF080" x="234.324856" y="178" width="98" height="7" rx="3.5"></rect>
                    <path d="M106.478856,139.884 L106.478856,144 L101.382856,144 L101.382856,124.96 L108.606856,124.96 C113.730856,124.96 116.614856,128.208 116.614856,132.436 C116.614856,136.664 113.730856,139.884 108.606856,139.884 L106.478856,139.884 Z M106.478856,135.572 L108.354856,135.572 C110.650856,135.572 111.490856,134.284 111.490856,132.436 C111.490856,130.588 110.650856,129.272 108.354856,129.272 L106.478856,129.272 L106.478856,135.572 Z M129.902856,144.252 C124.414856,144.252 119.682856,140.22 119.682856,134.48 C119.682856,128.74 124.414856,124.736 129.902856,124.736 C135.390856,124.736 140.122856,128.74 140.122856,134.48 C140.122856,140.22 135.390856,144.252 129.902856,144.252 Z M129.902856,139.828 C133.094856,139.828 134.914856,137.476 134.914856,134.48 C134.914856,131.484 133.094856,129.16 129.902856,129.16 C126.710856,129.16 124.890856,131.484 124.890856,134.48 C124.890856,137.476 126.710856,139.828 129.902856,139.828 Z M144.114856,144 L149.210856,144 L149.210856,124.96 L144.114856,124.96 L144.114856,144 Z M170.870856,144.336 C171.178856,144.588 171.262856,144.504 171.262856,144.28 L171.262856,124.96 L166.474856,124.96 L166.474856,134.508 L154.518856,124.652 C154.210856,124.4 154.126856,124.54 154.126856,124.708 L154.126856,144 L158.914856,144 L158.914856,134.508 L170.870856,144.336 Z M180.098856,144 L185.194856,144 L185.194856,129.244 L190.234856,129.244 L190.234856,124.96 L175.058856,124.96 L175.058856,129.244 L180.098856,129.244 L180.098856,144 Z M194.002856,139.324 C195.962856,139.324 197.726856,137.7 197.726856,135.6 C197.726856,133.472 195.962856,131.876 194.002856,131.876 C192.042856,131.876 190.278856,133.472 190.278856,135.6 C190.278856,137.7 192.042856,139.324 194.002856,139.324 Z M201.578856,144 L206.674856,144 L206.674856,137.952 L213.590856,137.952 L213.590856,133.668 L206.674856,133.668 L206.674856,129.244 L214.514856,129.244 L214.514856,124.96 L201.578856,124.96 L201.578856,144 Z M230.322856,138.596 L233.710856,144 L228.194856,144 L225.534856,139.632 L223.630856,139.632 L223.630856,144 L218.534856,144 L218.534856,124.96 L225.758856,124.96 C230.882856,124.96 233.766856,128.208 233.766856,132.436 C233.766856,135.18 232.562856,137.364 230.322856,138.596 Z M223.630856,135.572 L225.506856,135.572 C227.802856,135.572 228.642856,134.284 228.642856,132.436 C228.642856,130.588 227.802856,129.272 225.506856,129.272 L223.630856,129.272 L223.630856,135.572 Z M237.646856,144 L251.394856,144 L251.394856,139.716 L242.742856,139.716 L242.742856,137.728 L249.630856,137.728 L249.630856,133.668 L242.742856,133.668 L242.742856,129.244 L251.002856,129.244 L251.002856,124.96 L237.646856,124.96 L237.646856,144 Z M255.358856,144 L269.106856,144 L269.106856,139.716 L260.454856,139.716 L260.454856,137.728 L267.342856,137.728 L267.342856,133.668 L260.454856,133.668 L260.454856,129.244 L268.714856,129.244 L268.714856,124.96 L255.358856,124.96 L255.358856,144 Z" id="point•free" fill="#FFFFFF"></path>
                </g>
            </g>
        </g>
    </svg>
    """

public let facebookIconSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="17px" height="16px" viewBox="0 0 17 16" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Episodes" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Sign-up" transform="translate(-938.000000, -696.000000)" fill="#264BC1">
                <g id="Group-4" transform="translate(813.000000, 688.000000)">
                    <g id="fb-share" transform="translate(92.000000, 0.000000)">
                        <path d="M48.3571428,8 C48.6279775,8 48.8623501,8.09895734 49.0602678,8.296875 C49.2581855,8.49479266 49.3571428,8.72916531 49.3571428,9 L49.3571428,23 C49.3571428,23.2708347 49.2581855,23.5052073 49.0602678,23.703125 C48.8623501,23.9010427 48.6279775,24 48.3571428,24 L44.3571428,24 L44.3571428,17.78125 L46.3571428,17.78125 L46.6930803,15.4765625 L44.3571428,15.4765625 L44.3571428,13.6640625 C44.3571428,13.3307275 44.4417774,13.0755217 44.6110491,12.8984375 C44.7803207,12.7213533 45.0290162,12.6328125 45.3571428,12.6328125 L46.8571428,12.6328125 L46.8571428,10.5546875 C46.4092239,10.4817705 45.8623544,10.4453125 45.2165178,10.4453125 C44.1435958,10.4453125 43.3245935,10.7174452 42.7594866,11.2617188 C42.1943796,11.8059923 41.9118303,12.5468703 41.9118303,13.484375 L41.9118303,15.4765625 L39.9118303,15.4765625 L39.9118303,17.78125 L41.9118303,17.78125 L41.9118303,24 L34.3571428,24 C34.0863081,24 33.8519355,23.9010427 33.6540178,23.703125 C33.4561001,23.5052073 33.3571428,23.2708347 33.3571428,23 L33.3571428,9 C33.3571428,8.72916531 33.4561001,8.49479266 33.6540178,8.296875 C33.8519355,8.09895734 34.0863081,8 34.3571428,8 L48.3571428,8 Z" id="fb-icon"></path>
                    </g>
                </g>
            </g>
        </g>
    </svg>
    """

public let twitterIconSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="17px" height="14px" viewBox="0 0 17 14" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Episodes" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Sign-up" transform="translate(-846.000000, -697.000000)" fill="#4F85FF">
                <g id="Group-4" transform="translate(813.000000, 688.000000)">
                    <g id="tw-share">
                        <path d="M49.3571428,10.8671875 C48.9196406,11.5286491 48.3753752,12.0963518 47.7243303,12.5703125 L47.7243303,13 C47.7243303,13.7291703 47.6318833,14.4622359 47.4469866,15.1992188 C47.2620898,15.9362016 46.9964675,16.6523403 46.6501116,17.3476562 C46.3037557,18.0429722 45.8597497,18.6927053 45.3180803,19.296875 C44.7764109,19.9010447 44.1696462,20.4283832 43.4977678,20.8789062 C42.8258894,21.3294293 42.0511576,21.6848945 41.1735491,21.9453125 C40.2959405,22.2057305 39.3701685,22.3359375 38.3962053,22.3359375 C36.4222371,22.3359375 34.7425664,21.8906295 33.3571428,21 C33.5967273,21.0364585 33.8571414,21.0546875 34.1383928,21.0546875 C34.8675631,21.0546875 35.5810976,20.9140639 36.2790178,20.6328125 C36.976938,20.3515611 37.6201607,19.9739607 38.2087053,19.5 C37.5055768,19.4947916 36.8714685,19.2773459 36.3063616,18.8476562 C35.7412546,18.4179666 35.3571438,17.8750033 35.1540178,17.21875 C35.3727689,17.2604169 35.5758919,17.28125 35.7633928,17.28125 C36.0654776,17.28125 36.3571414,17.2421879 36.6383928,17.1640625 C35.8831807,17.0130201 35.2555828,16.6367218 34.7555803,16.0351562 C34.2555778,15.4335907 34.0055803,14.7369831 34.0055803,13.9453125 L34.0055803,13.90625 C34.4743326,14.1614596 34.9691194,14.2968749 35.4899553,14.3125 C35.036828,14.0104152 34.6787587,13.6197941 34.4157366,13.140625 C34.1527144,12.6614559 34.0212053,12.1458361 34.0212053,11.59375 C34.0212053,11.057289 34.1696413,10.5260443 34.4665178,10 C35.2894386,11.000005 36.2842203,11.7929658 37.4508928,12.3789062 C38.6175653,12.9648467 39.8753652,13.2942705 41.2243303,13.3671875 C41.1774551,13.1640615 41.1540178,12.9114598 41.1540178,12.609375 C41.1540178,11.7031205 41.473025,10.9309928 42.1110491,10.2929688 C42.7490731,9.65494473 43.5212008,9.3359375 44.4274553,9.3359375 C45.3597516,9.3359375 46.1644311,9.67968406 46.8415178,10.3671875 C47.5863132,10.2161451 48.2764105,9.95052273 48.9118303,9.5703125 C48.6618291,10.3619831 48.180063,10.9687479 47.4665178,11.390625 C48.1175627,11.317708 48.7477648,11.1432305 49.3571428,10.8671875 Z" id="tw-icon"></path>
                    </g>
                </g>
            </g>
        </g>
    </svg>
    """

public let pointFreePointersLogoSvgBase64 =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="448px" height="138px" viewBox="0 0 448 138" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <defs>
            <rect id="path-1" x="0" y="0" width="450" height="155"></rect>
        </defs>
        <g id="Pointers" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="pfp-logo">
                <mask id="mask-2" fill="white">
                    <use xlink:href="#path-1"></use>
                </mask>
                <use id="Mask" fill-opacity="0" fill="#D8D8D8" xlink:href="#path-1"></use>
                <path d="M200.988627,104.717114 L200.988627,104.73986 C200.988627,108.756086 197.732832,112.011881 193.716606,112.011881 C189.700379,112.011881 186.444584,108.756086 186.444584,104.73986 L186.444584,104.717114 C186.444584,100.700888 189.700379,97.4450926 193.716606,97.4450926 C197.732832,97.4450926 200.988627,100.700888 200.988627,104.717114 Z M131.068041,7.27202183 L131.068041,7.29476709 C131.068041,11.3109939 127.812245,14.5667889 123.796019,14.5667889 C119.779792,14.5667889 116.523997,11.3109939 116.523997,7.29476709 L116.523997,7.27202183 C116.523997,3.25579507 119.779792,5.79261824e-14 123.796019,5.81721054e-14 C127.812245,5.84180283e-14 131.068041,3.25579507 131.068041,7.27202183 Z M63.3358838,65.4595691 L195.909585,65.4595691 C198.720943,65.4595691 201,67.7386257 201,70.5499844 C201,73.3613431 198.720943,75.6403997 195.909585,75.6403997 L63.3358838,75.6403997 C60.5245251,75.6403997 58.2454685,73.3613431 58.2454685,70.5499844 C58.2454685,67.7386257 60.5245251,65.4595691 63.3358838,65.4595691 Z M5.46223258,100.365274 L138.035934,100.365274 C140.847292,100.365274 143.126349,102.64433 143.126349,105.455689 C143.126349,108.267048 140.847292,110.546104 138.035934,110.546104 L5.46223258,110.546104 C2.65087385,110.546104 0.371817298,108.267048 0.371817298,105.455689 C0.371817298,102.64433 2.65087385,100.365274 5.46223258,100.365274 Z M6.99861754,30.5538643 L139.572318,30.5538643 C142.383677,30.5538643 144.662734,32.8329209 144.662734,35.6442796 C144.662734,38.4556383 142.383677,40.7346949 139.572318,40.7346949 L6.99861754,40.7346949 C4.18725881,40.7346949 1.90820226,38.4556383 1.90820226,35.6442796 C1.90820226,32.8329209 4.18725881,30.5538643 6.99861754,30.5538643 Z" id="lines" fill="#FFFFFF" mask="url(#mask-2)"></path>
                <g id="Group-2" mask="url(#mask-2)" fill="#121212">
                    <g transform="translate(115.000000, 55.000000)" id="logo">
                        <path d="M7.23121318,21.5434056 L7.23121318,27.3127704 L0,27.3127704 L0,0.624552309 L10.2508407,0.624552309 C17.5217858,0.624552309 21.6141757,5.17724833 21.6141757,11.1036026 C21.6141757,17.0299569 17.5217858,21.5434056 10.2508407,21.5434056 L7.23121318,21.5434056 Z M7.23121318,15.4993091 L9.8932532,15.4993091 C13.1512723,15.4993091 14.3432305,13.6939297 14.3432305,11.1036026 C14.3432305,8.51327559 13.1512723,6.66864875 9.8932532,6.66864875 L7.23121318,6.66864875 L7.23121318,15.4993091 Z M40.4698195,27.6659968 C32.6823591,27.6659968 25.9676612,22.0143741 25.9676612,13.9686613 C25.9676612,5.92294854 32.6823591,0.310573273 40.4698195,0.310573273 C48.2572798,0.310573273 54.9719778,5.92294854 54.9719778,13.9686613 C54.9719778,22.0143741 48.2572798,27.6659968 40.4698195,27.6659968 Z M40.4698195,21.4649108 C44.9992607,21.4649108 47.5818368,18.1681309 47.5818368,13.9686613 C47.5818368,9.76919173 44.9992607,6.51165924 40.4698195,6.51165924 C35.9403782,6.51165924 33.3578021,9.76919173 33.3578021,13.9686613 C33.3578021,18.1681309 35.9403782,21.4649108 40.4698195,21.4649108 Z M60.6366173,27.3127704 L67.8678305,27.3127704 L67.8678305,0.624552309 L60.6366173,0.624552309 L60.6366173,27.3127704 Z M98.6033245,27.7837389 C99.0403759,28.1369653 99.1595717,28.0192232 99.1595717,27.7052442 L99.1595717,0.624552309 L92.3654098,0.624552309 L92.3654098,14.0079087 L75.3998712,0.192831135 C74.9628199,-0.160395281 74.843624,0.0358416168 74.843624,0.271325894 L74.843624,27.3127704 L81.6377859,27.3127704 L81.6377859,14.0079087 L98.6033245,27.7837389 Z M111.697837,27.3127704 L118.92905,27.3127704 L118.92905,6.62940137 L126.080799,6.62940137 L126.080799,0.624552309 L104.546088,0.624552309 L104.546088,6.62940137 L111.697837,6.62940137 L111.697837,27.3127704 Z M131.427583,20.758458 C134.208819,20.758458 136.711932,18.48211 136.711932,15.5385565 C136.711932,12.5557557 134.208819,10.318655 131.427583,10.318655 C128.646348,10.318655 126.143235,12.5557557 126.143235,15.5385565 C126.143235,18.48211 128.646348,20.758458 131.427583,20.758458 Z M142.177911,27.3127704 L149.409125,27.3127704 L149.409125,18.8353364 L159.222914,18.8353364 L159.222914,12.8304873 L149.409125,12.8304873 L149.409125,6.62940137 L160.534068,6.62940137 L160.534068,0.624552309 L142.177911,0.624552309 L142.177911,27.3127704 Z M182.965586,19.7380261 L187.773151,27.3127704 L179.945959,27.3127704 L176.171424,21.1901792 L173.469653,21.1901792 L173.469653,27.3127704 L166.238439,27.3127704 L166.238439,0.624552309 L176.48928,0.624552309 C183.760225,0.624552309 187.852615,5.17724833 187.852615,11.1036026 C187.852615,14.9498458 186.144142,18.0111414 182.965586,19.7380261 Z M173.469653,15.4993091 L176.131693,15.4993091 C179.389712,15.4993091 180.58167,13.6939297 180.58167,11.1036026 C180.58167,8.51327559 179.389712,6.66864875 176.131693,6.66864875 L173.469653,6.66864875 L173.469653,15.4993091 Z M193.358327,27.3127704 L212.86671,27.3127704 L212.86671,21.3079213 L200.58954,21.3079213 L200.58954,18.5213574 L210.363597,18.5213574 L210.363597,12.8304873 L200.58954,12.8304873 L200.58954,6.62940137 L212.310462,6.62940137 L212.310462,0.624552309 L193.358327,0.624552309 L193.358327,27.3127704 Z M218.491617,27.3127704 L238,27.3127704 L238,21.3079213 L225.72283,21.3079213 L225.72283,18.5213574 L235.496888,18.5213574 L235.496888,12.8304873 L225.72283,12.8304873 L225.72283,6.62940137 L237.443753,6.62940137 L237.443753,0.624552309 L218.491617,0.624552309 L218.491617,27.3127704 Z"></path>
                    </g>
                </g>
                <path d="M298.626354,109.628474 C298.649791,109.534724 298.702525,109.493709 298.784557,109.505428 C298.866589,109.517146 298.94276,109.511287 299.013073,109.487849 C298.825572,109.300349 298.708385,109.048398 298.66151,108.73199 C298.614635,108.415582 298.497449,108.151913 298.309948,107.940974 C298.356823,107.823786 298.397838,107.7066 298.432994,107.589412 C298.468151,107.472224 298.567759,107.425349 298.731823,107.448787 C298.66151,106.933159 298.673228,106.452695 298.766979,106.007381 C298.860729,105.562066 299.007212,105.140195 299.206432,104.741756 C299.405652,104.343316 299.640024,103.950742 299.909557,103.564021 C300.17909,103.177301 300.442759,102.784726 300.700573,102.386287 C300.911511,102.339412 301.040416,102.35699 301.087291,102.439021 C301.134167,102.521053 301.216197,102.503475 301.333385,102.386287 C301.708387,102.128473 301.966197,101.841367 302.106823,101.524959 C302.247448,101.208551 302.31776,100.862851 302.31776,100.487849 C302.528699,99.8784714 302.768931,99.2749618 303.038463,98.6773026 C303.307996,98.0796433 303.61854,97.5230083 303.970104,97.0073807 C304.321668,96.4917531 304.696664,96.0405858 305.095104,95.6538651 C305.493543,95.2671444 305.927133,94.9683193 306.395885,94.7573807 C306.724012,94.4761293 307.140023,94.2183194 307.643932,93.9839432 C308.147841,93.749567 308.686898,93.5679282 309.261119,93.4390213 C309.835341,93.3101144 310.427132,93.2398026 311.03651,93.2280838 C311.645888,93.216365 312.208383,93.2808175 312.72401,93.4214432 C312.841198,93.4683184 312.91151,93.4390219 312.934948,93.3335526 C312.958385,93.2280833 313.016978,93.1753495 313.110729,93.1753495 C314.28261,93.3159752 315.331427,93.5503478 316.257213,93.8784745 C317.182999,94.2066011 317.974007,94.6636278 318.63026,95.2495682 C319.075575,95.9526967 319.497446,96.7085485 319.895885,97.5171463 C320.294325,98.3257441 320.604868,99.1694857 320.827526,100.048396 C321.050183,100.927307 321.149791,101.835501 321.126354,102.773006 C321.102916,103.71051 320.903699,104.648001 320.528698,105.585506 C319.708381,106.616761 318.712297,107.442924 317.540416,108.064021 C316.368536,108.685118 315.108782,109.200738 313.761119,109.610896 C312.413457,110.021055 311.018939,110.355036 309.577526,110.612849 C308.136112,110.870663 306.735736,111.15191 305.376354,111.456599 C305.399791,112.089415 305.399791,112.763237 305.376354,113.478084 C305.352916,114.192931 305.276745,114.907768 305.147838,115.622615 C305.018931,116.337462 304.831433,117.034721 304.585338,117.714412 C304.339243,118.394103 304.005262,119.026909 303.583385,119.612849 C303.114633,119.589412 302.62831,119.636286 302.124401,119.753474 C301.620492,119.870663 301.169325,119.987849 300.770885,120.105037 C300.583384,119.987849 300.401745,119.876522 300.225963,119.771053 C300.050181,119.665583 299.833386,119.601131 299.575573,119.577693 C299.645885,119.296442 299.581433,119.003476 299.382213,118.698787 C299.182994,118.394098 298.983777,118.095273 298.784557,117.802303 C298.585337,117.509332 298.456432,117.216367 298.397838,116.923396 C298.339244,116.630426 298.474008,116.366757 298.802135,116.132381 C298.614634,115.968317 298.520885,115.710507 298.520885,115.358943 C298.614636,115.452694 298.731822,115.499568 298.872448,115.499568 C298.825572,115.077691 298.878306,114.57379 299.030651,113.987849 C299.182995,113.401909 299.3119,112.821837 299.417369,112.247615 C299.522839,111.673393 299.540417,111.146055 299.470104,110.665584 C299.399791,110.185113 299.118544,109.839413 298.626354,109.628474 Z M307.766979,101.401912 C307.673228,101.659726 307.597057,101.888239 307.538463,102.087459 C307.479869,102.286679 307.415417,102.468317 307.345104,102.632381 C307.274791,102.796444 307.19862,102.960505 307.116588,103.124568 C307.034557,103.288632 306.946667,103.47613 306.852916,103.687068 C306.899792,103.827694 306.993541,103.933162 307.134166,104.003474 C307.274792,104.073787 307.368541,104.179255 307.415416,104.319881 C307.743543,104.155817 308.071665,104.032772 308.399791,103.95074 C308.727918,103.868708 309.05604,103.804256 309.384166,103.757381 C309.712293,103.710505 310.040415,103.651912 310.368541,103.581599 C310.696668,103.511287 311.036508,103.3941 311.388073,103.230037 C311.739637,103.065974 312.108774,102.931209 312.495494,102.82574 C312.882215,102.720271 313.263071,102.608944 313.638073,102.491756 C314.013074,102.374568 314.376352,102.239803 314.727916,102.087459 C315.079481,101.935114 315.384165,101.730038 315.641979,101.472224 C315.313852,101.167535 314.82167,100.898007 314.165416,100.663631 C313.509163,100.429255 312.800186,100.282772 312.038463,100.224178 C311.276741,100.165584 310.51503,100.218317 309.753307,100.382381 C308.991584,100.546444 308.329482,100.886284 307.766979,101.401912 Z M340.56776,108.995662 C340.544323,109.276913 340.56776,109.546442 340.638073,109.804256 C340.708385,110.062069 340.796275,110.31402 340.901744,110.560115 C341.007214,110.80621 341.106822,111.058161 341.200573,111.315974 C341.294323,111.573788 341.364635,111.831598 341.41151,112.089412 C341.458385,112.417539 341.458385,112.774957 341.41151,113.161678 C341.364635,113.548398 341.282604,113.923395 341.165416,114.286678 C341.048228,114.649961 340.895886,114.978082 340.708385,115.271053 C340.520884,115.564023 340.29823,115.780817 340.040416,115.921443 C339.71229,116.34332 339.313856,116.671442 338.845104,116.905818 C338.376352,117.140194 337.942762,117.421442 337.544323,117.749568 C337.286509,117.913632 336.999402,118.030818 336.682994,118.101131 C336.366587,118.171444 336.044324,118.241755 335.716198,118.312068 C335.458384,118.452694 335.153699,118.599177 334.802135,118.751521 C334.450571,118.903866 334.087293,119.044489 333.712291,119.173396 C333.337289,119.302303 332.962293,119.396052 332.587291,119.454646 C332.212289,119.51324 331.884168,119.530818 331.602916,119.507381 C330.899788,119.765194 330.214248,119.935115 329.546276,120.017146 C328.878304,120.099178 328.099015,120.046444 327.208385,119.858943 C326.364631,119.976131 325.585342,119.940975 324.870494,119.753474 C324.155647,119.565974 323.522841,119.267148 322.972057,118.85699 C322.421273,118.446832 321.940809,117.954649 321.530651,117.380428 C321.120492,116.806206 320.786511,116.202696 320.528698,115.569881 C320.341197,114.233937 320.388071,113.021058 320.669323,111.931209 C320.950574,110.84136 321.354867,109.833557 321.882213,108.907771 C322.40956,107.981985 323.013069,107.126525 323.69276,106.341365 C324.372451,105.556205 325.028694,104.788634 325.66151,104.038631 C325.802136,104.085506 325.913463,104.079647 325.995494,104.021053 C326.077526,103.962459 326.153697,103.880428 326.22401,103.774959 C326.294323,103.66949 326.370494,103.569881 326.452526,103.476131 C326.534557,103.38238 326.634166,103.323787 326.751354,103.300349 C326.915417,102.948785 327.155649,102.685116 327.472057,102.509334 C327.788465,102.333552 328.040415,102.081601 328.227916,101.753474 C328.438855,101.706599 328.614634,101.636287 328.75526,101.542537 C328.895886,101.448786 329.03065,101.343319 329.159557,101.226131 C329.288464,101.108943 329.423228,100.997616 329.563854,100.892146 C329.70448,100.786677 329.86854,100.710506 330.056041,100.663631 L330.056041,100.276912 C330.477918,100.089411 330.788462,99.8726163 330.987682,99.6265213 C331.186902,99.3804263 331.427134,99.0816012 331.708385,98.730037 C331.919324,98.7065993 332.089244,98.6948807 332.218151,98.6948807 C332.347058,98.6948807 332.540415,98.6714434 332.798229,98.6245682 C332.98573,98.2964416 333.15565,98.0503503 333.307994,97.886287 C333.460339,97.7222236 333.688852,97.5464441 333.993541,97.3589432 C334.274793,97.4995689 334.585337,97.593318 334.925182,97.6401932 C335.265027,97.6870684 335.563853,97.8394107 335.821666,98.0972245 C335.845104,98.3316006 335.915416,98.5132394 336.032604,98.6421463 C336.149792,98.7710532 336.278697,98.8882396 336.419323,98.9937088 C336.559948,99.0991781 336.694713,99.2105051 336.823619,99.3276932 C336.952526,99.4448813 337.040416,99.6089421 337.087291,99.8198807 C337.181042,99.8433183 337.245494,99.8315997 337.280651,99.7847245 C337.315807,99.7378492 337.38026,99.7261306 337.47401,99.7495682 C337.544323,100.077695 337.626353,100.29449 337.720104,100.399959 C337.813854,100.505428 337.907603,100.710504 338.001354,101.015193 C337.884166,101.249569 337.796276,101.489801 337.737682,101.735896 C337.679088,101.981991 337.532605,102.16363 337.298229,102.280818 C337.274791,102.444882 337.321666,102.532771 337.438854,102.54449 C337.556042,102.556209 337.626354,102.620661 337.649791,102.737849 C337.673229,102.948788 337.638073,103.10113 337.544323,103.194881 C338.106825,104.108948 338.610727,105.075735 339.056041,106.095271 C339.501356,107.114808 340.005257,108.081595 340.56776,108.995662 Z M336.173229,112.335506 C336.266979,112.077692 336.390025,111.866757 336.542369,111.702693 C336.694714,111.53863 336.782604,111.292539 336.806041,110.964412 C336.688853,110.776911 336.618541,110.630428 336.595104,110.524959 C336.571666,110.41949 336.559948,110.249569 336.559948,110.015193 C336.04432,109.288627 335.599012,108.509338 335.22401,107.677303 C334.849008,106.845267 334.427137,106.065978 333.958385,105.339412 C333.91151,105.433162 333.829479,105.526911 333.712291,105.620662 C333.595103,105.714412 333.466198,105.690975 333.325573,105.550349 C333.278697,105.73785 333.196667,105.872615 333.079479,105.954646 C332.962291,106.036678 332.833386,106.071834 332.69276,106.060115 C332.552134,106.048396 332.405652,106.01324 332.253307,105.954646 C332.100962,105.896052 331.966198,105.819881 331.84901,105.726131 C331.169319,106.265196 330.489639,106.792534 329.809948,107.308162 C329.130257,107.82379 328.520888,108.42144 327.981823,109.101131 C327.888072,109.148006 327.829479,109.165584 327.806041,109.153865 C327.782604,109.142146 327.712292,109.159724 327.595104,109.206599 C327.407603,109.651914 327.17909,110.056207 326.909557,110.41949 C326.640024,110.782773 326.376355,111.15191 326.118541,111.526912 C325.860728,111.901914 325.638074,112.300347 325.450573,112.722224 C325.263072,113.144102 325.157604,113.636284 325.134166,114.198787 C325.532606,114.526914 326.001351,114.72613 326.540416,114.796443 C327.079482,114.866756 327.636117,114.878475 328.210338,114.831599 C328.78456,114.784724 329.347054,114.708553 329.897838,114.603084 C330.448622,114.497615 330.958383,114.444881 331.427135,114.444881 C332.411515,114.210505 333.255257,113.899961 333.958385,113.51324 C334.661514,113.126519 335.399788,112.733945 336.173229,112.335506 Z M350.165416,117.081599 C350.282604,117.784728 350.241589,118.335504 350.042369,118.733943 C349.84315,119.132383 349.561903,119.431208 349.198619,119.630428 C348.835336,119.829647 348.419325,119.940974 347.950573,119.964412 C347.48182,119.98785 347.036512,119.964412 346.614635,119.894099 C346.380259,119.565973 346.169324,119.155821 345.981823,118.663631 C345.794322,118.171441 345.618542,117.66168 345.454479,117.134334 C345.290416,116.606987 345.143933,116.085508 345.015026,115.569881 C344.886119,115.054253 344.774792,114.597226 344.681041,114.198787 C344.798229,113.917536 344.874401,113.595273 344.909557,113.23199 C344.944713,112.868707 344.968151,112.49957 344.979869,112.124568 C344.991588,111.749566 345.003307,111.368711 345.015026,110.98199 C345.026745,110.595269 345.056041,110.226132 345.102916,109.874568 C345.26698,108.890188 345.507212,107.952698 345.823619,107.062068 C346.140027,106.171439 346.391978,105.245667 346.579479,104.284724 C346.626354,104.073786 346.608776,103.874569 346.526744,103.687068 C346.444713,103.499567 346.427135,103.323788 346.47401,103.159724 C346.497448,103.065974 346.56776,103.001522 346.684948,102.966365 C346.802136,102.931209 346.872448,102.855038 346.895885,102.737849 C346.966198,102.480036 346.983776,102.2691 346.948619,102.105037 C346.913463,101.940974 346.954478,101.741757 347.071666,101.507381 C347.188854,101.31988 347.335337,101.044492 347.511119,100.681209 C347.686902,100.317926 347.751354,99.9370702 347.704479,99.5386307 C347.915417,99.398005 348.114634,99.2339441 348.302135,99.0464432 C348.489636,98.8589423 348.682994,98.6890221 348.882213,98.5366776 C349.081433,98.3843331 349.304087,98.2495688 349.550182,98.1323807 C349.796277,98.0151926 350.095102,97.9565995 350.446666,97.9565995 C350.634167,98.1909756 350.757213,98.4487855 350.815807,98.730037 L350.991588,99.573787 C351.050182,99.8550384 351.138072,100.118708 351.25526,100.364803 C351.372448,100.610898 351.571665,100.804255 351.852916,100.944881 L351.852916,101.648006 C351.852916,101.929257 351.864635,102.204645 351.888073,102.474178 C351.91151,102.74371 351.964244,102.989801 352.046276,103.212459 C352.128307,103.435116 352.263072,103.581599 352.450573,103.651912 C352.028696,104.776918 351.636121,105.890188 351.272838,106.991756 C350.909555,108.093324 350.61073,109.200735 350.376354,110.314021 C350.141978,111.427308 349.995495,112.546438 349.936901,113.671443 C349.878307,114.796449 349.954478,115.933156 350.165416,117.081599 Z M361.450573,113.144099 C361.239634,113.190975 361.087292,113.31402 360.993541,113.51324 C360.899791,113.71246 360.811901,113.917536 360.729869,114.128474 C360.647838,114.339413 360.565807,114.53277 360.483776,114.708553 C360.401744,114.884335 360.278698,114.972224 360.114635,114.972224 C360.091198,115.159725 360.026745,115.323786 359.921276,115.464412 C359.815806,115.605038 359.716198,115.751521 359.622448,115.903865 C359.528697,116.05621 359.458385,116.22027 359.41151,116.396053 C359.364635,116.571835 359.388072,116.776911 359.481823,117.011287 C359.294322,117.151913 359.177135,117.351129 359.13026,117.608943 C359.083385,117.866757 359.030651,118.112848 358.972057,118.347224 C358.913463,118.581601 358.813855,118.780817 358.673229,118.944881 C358.532603,119.108944 358.274793,119.179256 357.899791,119.155818 C357.829479,119.249569 357.81776,119.308162 357.864635,119.331599 C357.91151,119.355037 357.946666,119.437067 357.970104,119.577693 C357.71229,119.530818 357.518933,119.548396 357.390026,119.630428 C357.261119,119.712459 357.102917,119.765193 356.915416,119.788631 C356.470102,119.648005 356.100965,119.401914 355.807994,119.050349 C355.515024,118.698785 355.257214,118.306211 355.034557,117.872615 C354.8119,117.439019 354.612683,116.99957 354.436901,116.554256 C354.261119,116.108941 354.091198,115.722226 353.927135,115.394099 C353.997448,115.136286 354.009167,114.884335 353.962291,114.63824 C353.915416,114.392145 353.833386,114.187069 353.716198,114.023006 C354.864641,111.585494 356.118535,108.983957 357.477916,106.218318 C358.837298,103.452679 360.126348,100.722238 361.345104,98.026912 C361.884169,98.2378505 362.417367,98.4780825 362.944713,98.7476151 C363.47206,99.0171477 363.952524,99.3276914 364.386119,99.6792557 C364.819715,100.03082 365.188852,100.435113 365.493541,100.892146 C365.79823,101.34918 365.997447,101.85894 366.091198,102.421443 C366.044322,102.562069 365.991589,102.614803 365.932994,102.579646 C365.8744,102.54449 365.821667,102.562068 365.774791,102.632381 C365.868542,102.749569 365.921276,102.831599 365.932994,102.878474 C365.944713,102.92535 365.944713,102.972224 365.932994,103.019099 C365.921276,103.065975 365.915416,103.130427 365.915416,103.212459 C365.915416,103.29449 365.950572,103.405817 366.020885,103.546443 C365.927135,103.523006 365.862682,103.534724 365.827526,103.581599 C365.792369,103.628475 365.716198,103.628475 365.59901,103.581599 C365.692761,103.839413 365.722057,104.050349 365.686901,104.214412 C365.651744,104.378475 365.59901,104.524958 365.528698,104.653865 C365.458385,104.782772 365.399792,104.923396 365.352916,105.07574 C365.306041,105.228085 365.31776,105.421442 365.388073,105.655818 C365.31776,105.655818 365.265026,105.673396 365.229869,105.708553 C365.194713,105.743709 365.165416,105.778865 365.141979,105.814021 C365.118541,105.849178 365.089245,105.884334 365.054088,105.91949 C365.018932,105.954647 364.966198,105.960506 364.895885,105.937068 C364.966198,106.452696 364.989635,106.968316 364.966198,107.483943 C364.94276,107.999571 364.925182,108.52105 364.913463,109.048396 C364.901744,109.575743 364.907604,110.126518 364.931041,110.70074 C364.954479,111.274962 365.036509,111.866753 365.177135,112.476131 C366.020889,111.983941 366.694711,111.351135 367.198619,110.577693 C367.702528,109.804252 368.11268,108.937073 368.429088,107.976131 C368.745496,107.015188 368.985728,105.983949 369.149791,104.882381 C369.313855,103.780813 369.489634,102.655824 369.677135,101.507381 C369.888074,101.296442 370.052135,101.038632 370.169323,100.733943 C370.286511,100.429254 370.391978,100.106992 370.485729,99.7671463 L370.766979,98.7476151 C370.860729,98.4077696 370.977916,98.0972259 371.118541,97.8159745 C371.470106,97.8628497 371.768931,97.9741767 372.015026,98.1499588 C372.261121,98.325741 372.489634,98.5132391 372.700573,98.7124588 C372.911511,98.9116786 373.134165,99.0991767 373.368541,99.2749588 C373.602918,99.450741 373.895883,99.562068 374.247448,99.6089432 C374.528699,100.007383 374.786509,100.394098 375.020885,100.769099 C375.255261,101.144101 375.407604,101.542535 375.477916,101.964412 C375.360728,102.151913 375.272839,102.362848 375.214244,102.597224 C375.15565,102.831601 375.091198,103.065973 375.020885,103.300349 C374.950572,103.534726 374.856823,103.75152 374.739635,103.95074 C374.622447,104.14996 374.458386,104.31988 374.247448,104.460506 C374.177135,105.069884 374.089245,105.626519 373.983776,106.130428 C373.878306,106.634336 373.772839,107.132378 373.667369,107.624568 C373.5619,108.116758 373.47401,108.626519 373.403698,109.153865 C373.333385,109.681211 373.309948,110.261284 373.333385,110.894099 C373.145884,111.433165 372.946667,112.007378 372.735729,112.616756 C372.52479,113.226134 372.360729,113.776909 372.243541,114.269099 C371.868539,114.573788 371.575574,114.937066 371.364635,115.358943 C371.153697,115.78082 370.931043,116.202691 370.696666,116.624568 C370.602916,116.718319 370.544323,116.741756 370.520885,116.694881 C370.497448,116.648005 370.415417,116.61285 370.274791,116.589412 C370.298229,116.683162 370.298229,116.753474 370.274791,116.800349 C370.251354,116.847225 370.222057,116.88824 370.186901,116.923396 C370.151744,116.958553 370.128307,117.011287 370.116588,117.081599 C370.104869,117.151912 370.122447,117.245661 370.169323,117.362849 C369.841196,117.503475 369.54823,117.702692 369.290416,117.960506 C369.032603,118.218319 368.757215,118.452692 368.464244,118.663631 C368.171274,118.874569 367.819715,119.015193 367.409557,119.085506 C366.999399,119.155819 366.466201,119.073788 365.809948,118.839412 C364.849005,118.136283 363.999404,117.327698 363.261119,116.413631 C362.522835,115.499564 361.919325,114.409731 361.450573,113.144099 Z M394.56776,102.632381 C393.98182,102.632381 393.349014,102.673396 392.669323,102.755428 C391.989632,102.837459 391.32167,102.925349 390.665416,103.019099 C390.196664,103.089412 389.745497,103.148005 389.311901,103.194881 C388.878305,103.241756 388.485731,103.276912 388.134166,103.300349 C387.970103,103.815977 387.806042,104.273004 387.641979,104.671443 C387.477916,105.069883 387.302136,105.456598 387.114635,105.831599 C386.927134,106.230039 386.757214,106.634332 386.604869,107.04449 C386.452525,107.454648 386.306042,107.894097 386.165416,108.362849 C385.93104,108.737851 385.737683,109.083551 385.585338,109.399959 C385.432994,109.716367 385.309948,110.073785 385.216198,110.472224 C384.934946,110.823789 384.724011,111.192926 384.583385,111.579646 C384.442759,111.966367 384.337292,112.300348 384.266979,112.581599 L384.126354,113.003474 C383.985728,113.1441 383.903698,113.255427 383.88026,113.337459 C383.856823,113.41949 383.798229,113.495662 383.704479,113.565974 C383.681041,113.823788 383.599011,114.081598 383.458385,114.339412 C383.41151,114.4566 383.376354,114.567927 383.352916,114.673396 C383.329479,114.778866 383.31776,114.855037 383.31776,114.901912 C383.31776,115.136288 383.259167,115.28863 383.141979,115.358943 C383.165416,115.476131 383.165416,115.599177 383.141979,115.728084 C383.118541,115.856991 383.095104,115.980036 383.071666,116.097224 C383.001354,116.378476 382.977916,116.60113 383.001354,116.765193 C383.071667,116.858944 383.136119,116.940974 383.194713,117.011287 C383.253307,117.0816 383.294323,117.175349 383.31776,117.292537 C383.458386,117.386287 383.563854,117.444881 383.634166,117.468318 C383.727917,117.538631 383.86854,117.620661 384.056041,117.714412 L384.372448,118.065974 C384.489636,118.745665 384.384168,119.284722 384.056041,119.683162 C383.704477,120.128477 383.106827,120.351131 382.263073,120.351131 L381.841198,120.351131 C381.72401,120.233943 381.653698,120.140194 381.63026,120.069881 C381.442759,119.999568 381.290417,119.940975 381.173229,119.894099 C381.056041,119.847224 380.950573,119.753475 380.856823,119.612849 C380.552134,119.472224 380.306042,119.355037 380.118541,119.261287 L379.802135,119.085506 L379.802135,118.663631 C379.638072,118.452692 379.515026,118.218319 379.432994,117.960506 C379.350963,117.702692 379.28651,117.444882 379.239635,117.187068 C379.19276,116.97613 379.140026,116.776913 379.081432,116.589412 C379.022838,116.401911 378.946667,116.23785 378.852916,116.097224 C378.806041,114.737843 378.934946,113.589417 379.239635,112.651912 C379.544324,111.714407 379.860727,110.812073 380.188854,109.944881 C380.376355,109.476128 380.563853,109.001524 380.751354,108.521053 C380.938855,108.040581 381.102916,107.54254 381.243541,107.026912 L381.559948,106.675349 C381.724011,106.300348 381.858775,105.989804 381.964244,105.743709 C382.069714,105.497614 382.181041,105.245663 382.298229,104.987849 C382.368542,104.823786 382.432994,104.665585 382.491588,104.51324 C382.550182,104.360896 382.602916,104.226131 382.649791,104.108943 C382.509166,104.108943 382.380261,104.114803 382.263073,104.126521 C382.145885,104.13824 382.028698,104.155818 381.91151,104.179256 C381.770884,104.202693 381.641979,104.220271 381.524791,104.23199 C381.407603,104.243709 381.290417,104.249568 381.173229,104.249568 C381.009166,104.249568 380.891979,104.23785 380.821666,104.214412 C380.563853,104.3316 380.282605,104.390193 379.977916,104.390193 L379.626354,104.390193 C379.556041,104.390193 379.491589,104.384334 379.432994,104.372615 C379.3744,104.360896 379.321667,104.355037 379.274791,104.355037 C379.040415,104.355037 378.852917,104.390193 378.712291,104.460506 L377.552135,104.460506 C377.50526,104.460506 377.446667,104.466365 377.376354,104.478084 C377.306041,104.489803 377.247448,104.495662 377.200573,104.495662 C377.036509,104.495662 376.86073,104.478084 376.673229,104.442928 C376.485728,104.407771 376.286511,104.3316 376.075573,104.214412 C375.794321,103.83941 375.606823,103.640193 375.513073,103.616756 L375.196666,103.194881 C375.173229,103.054255 375.138073,102.919491 375.091198,102.790584 C375.044322,102.661677 374.997448,102.515194 374.950573,102.351131 C374.856822,102.116755 374.768932,101.864804 374.686901,101.595271 C374.604869,101.325739 374.563854,101.003476 374.563854,100.628474 C374.587291,100.440974 374.634166,100.30035 374.704479,100.206599 C374.774792,100.112849 374.833385,100.030819 374.88026,99.9605057 L374.985729,99.433162 L375.44276,99.4683182 L376.427135,99.4683182 C377.177139,99.4683182 377.93885,99.4390216 378.712291,99.3804276 C379.485733,99.3218335 380.2826,99.2573811 381.102916,99.1870682 C381.571669,99.140193 382.046273,99.0933184 382.526744,99.0464432 C383.007216,98.999568 383.505258,98.9526934 384.020885,98.9058182 C384.114636,98.6948796 384.208385,98.4956629 384.302135,98.308162 L384.653698,98.1323807 C385.075575,98.1089431 385.3744,98.1616769 385.550182,98.2905838 C385.725964,98.4194907 385.872447,98.5425364 385.989635,98.6597245 C386.505263,98.6128492 387.026742,98.5601154 387.554088,98.5015213 C388.081435,98.4429273 388.614632,98.3667562 389.153698,98.2730057 C389.950577,98.1558176 390.753303,98.0503499 391.561901,97.9565995 C392.370499,97.862849 393.149788,97.8159745 393.899791,97.8159745 C394.227918,97.8159745 394.544321,97.8276931 394.84901,97.8511307 C395.153699,97.8745683 395.434946,97.9097242 395.69276,97.9565995 L396.044323,98.2026932 C396.114635,98.3433189 396.214244,98.4722239 396.343151,98.589412 C396.472058,98.7066 396.606822,98.8237864 396.747448,98.9409745 C397.028699,99.1987882 397.309946,99.5034727 397.591198,99.855037 C397.872449,100.206601 397.977917,100.722221 397.907604,101.401912 C397.720103,101.683163 397.579479,101.847224 397.485729,101.894099 L397.169323,102.315974 L396.81776,102.280818 C396.677134,102.280818 396.548229,102.298396 396.431041,102.333553 C396.313853,102.368709 396.196667,102.409724 396.079479,102.456599 C395.985728,102.503475 395.891979,102.54449 395.798229,102.579646 C395.704478,102.614803 395.599011,102.644099 395.481823,102.667537 C395.270884,102.667537 395.100964,102.661678 394.972057,102.649959 C394.84315,102.63824 394.708386,102.632381 394.56776,102.632381 Z M411.44276,99.7495682 C411.466198,99.9136315 411.472057,100.036677 411.460338,100.118709 C411.448619,100.20074 411.436901,100.276912 411.425182,100.347224 C411.413463,100.417537 411.407604,100.493708 411.407604,100.57574 C411.407604,100.657772 411.431041,100.780817 411.477916,100.944881 C410.985726,101.74176 410.417373,102.286676 409.772838,102.579646 C409.128304,102.872617 408.431045,103.083552 407.681041,103.212459 C406.931038,103.341366 406.163467,103.47613 405.378307,103.616756 C404.593147,103.757381 403.813858,104.062066 403.040416,104.530818 C402.735727,104.41363 402.460339,104.41363 402.214244,104.530818 C401.96815,104.648006 401.610731,104.683162 401.141979,104.636287 C401.001353,104.776913 400.825574,104.876521 400.614635,104.935115 C400.403697,104.993709 400.192761,105.058162 399.981823,105.128474 C400.075573,105.339413 400.116588,105.556208 400.104869,105.778865 C400.093151,106.001522 400.122447,106.183161 400.19276,106.323787 C401.013077,106.3941 401.921271,106.382381 402.917369,106.288631 C403.913468,106.19488 404.891974,106.136287 405.852916,106.112849 C405.876354,106.230038 405.864635,106.306209 405.81776,106.341365 C405.770885,106.376522 405.759166,106.440974 405.782604,106.534724 C405.946667,106.651913 406.110728,106.798395 406.274791,106.974178 C406.438855,107.14996 406.602916,107.31988 406.766979,107.483943 C406.931042,107.648007 407.100962,107.782771 407.276744,107.88824 C407.452527,107.993709 407.657603,108.034725 407.891979,108.011287 C407.985729,108.198788 408.050182,108.403864 408.085338,108.626521 C408.120495,108.849179 408.091198,109.077692 407.997448,109.312068 C407.622446,109.569882 407.247449,109.833551 406.872448,110.103084 C406.497446,110.372616 406.104872,110.612848 405.694713,110.823787 C405.284555,111.034726 404.850966,111.204646 404.393932,111.333553 C403.936898,111.462459 403.427138,111.515193 402.864635,111.491756 L402.600963,111.491756 C402.542369,111.491756 402.466198,111.4566 402.372448,111.386287 C402.184947,111.503475 401.921277,111.538631 401.581432,111.491756 C401.241587,111.44488 400.977917,111.480036 400.790416,111.597224 C400.532603,111.480036 400.222059,111.427303 399.858776,111.439021 C399.495493,111.45074 399.17323,111.409725 398.891979,111.315974 C398.516977,111.808164 398.177137,112.335503 397.872448,112.898006 C397.567759,113.460509 397.29823,114.058159 397.063854,114.690974 C397.860733,114.808163 398.522836,114.855037 399.050182,114.831599 C399.577528,114.808162 400.07557,114.767147 400.544323,114.708553 C401.013075,114.649959 401.493539,114.591365 401.985729,114.532771 C402.477919,114.474177 403.099006,114.456599 403.84901,114.480037 C404.317762,114.222223 404.874398,114.052303 405.518932,113.970271 C406.163466,113.88824 406.755257,113.753475 407.294323,113.565974 C407.481824,113.636287 407.645884,113.759333 407.78651,113.935115 C407.927136,114.110897 408.0619,114.274958 408.190807,114.427303 C408.319714,114.579647 408.454478,114.714411 408.595104,114.831599 C408.73573,114.948788 408.911509,114.995662 409.122448,114.972224 C409.216198,115.065975 409.280651,115.177302 409.315807,115.306209 C409.350963,115.435116 409.38026,115.558161 409.403698,115.675349 C409.427135,115.792538 409.46815,115.892146 409.526744,115.974178 C409.585339,116.056209 409.684947,116.085506 409.825573,116.062068 C409.474008,116.83551 409.069716,117.427301 408.612682,117.837459 C408.155648,118.247617 407.634169,118.558161 407.048229,118.769099 C406.462288,118.980038 405.811904,119.144099 405.097057,119.261287 C404.38221,119.378475 403.591202,119.519099 402.72401,119.683162 C402.513072,119.730037 402.337292,119.724178 402.196666,119.665584 C402.056041,119.60699 401.89198,119.577693 401.704479,119.577693 C401.165414,119.694881 400.520889,119.800349 399.770885,119.894099 C399.020881,119.98785 398.25917,120.023006 397.485729,119.999568 C396.712288,119.976131 395.974014,119.876522 395.270885,119.70074 C394.567757,119.524958 393.993544,119.214414 393.548229,118.769099 C393.360728,118.605036 393.237682,118.411679 393.179088,118.189021 C393.120494,117.966364 392.962293,117.796444 392.704479,117.679256 C392.681041,117.374567 392.634167,117.034726 392.563854,116.659724 C392.493541,116.284723 392.423229,115.892148 392.352916,115.48199 C392.282604,115.071832 392.22987,114.667539 392.194713,114.269099 C392.159557,113.87066 392.177135,113.507382 392.247448,113.179256 C392.31776,112.827691 392.464243,112.511288 392.686901,112.230037 C392.909558,111.948786 392.985729,111.644101 392.915416,111.315974 C393.032604,111.292537 393.102916,111.228084 393.126354,111.122615 C393.149791,111.017146 393.243541,110.976131 393.407604,110.999568 C393.54823,110.577691 393.700572,110.179258 393.864635,109.804256 C394.028698,109.429254 394.198619,109.065976 394.374401,108.714412 C394.550183,108.362848 394.725962,108.005429 394.901744,107.642146 C395.077527,107.278863 395.247447,106.87457 395.41151,106.429256 C395.083383,106.124567 394.843152,105.790586 394.690807,105.427303 C394.538462,105.06402 394.427135,104.700742 394.356823,104.337459 C394.28651,103.974176 394.227917,103.59918 394.181041,103.212459 C394.134166,102.825738 394.052136,102.433164 393.934948,102.034724 C394.169324,101.68316 394.45643,101.396054 394.796276,101.173396 C395.136121,100.950739 395.505258,100.7691 395.903698,100.628474 C396.302137,100.487849 396.735727,100.370662 397.204479,100.276912 C397.673231,100.183161 398.165414,100.077694 398.681041,99.9605057 C399.618546,99.6558167 400.456428,99.421444 401.194713,99.2573807 C401.932998,99.0933174 402.659553,98.9585531 403.374401,98.8530838 C404.089248,98.7476146 404.833381,98.6480062 405.606823,98.5542557 C406.380264,98.4605052 407.259161,98.3198816 408.243541,98.1323807 C408.45448,98.3433193 408.653697,98.448787 408.841198,98.448787 C408.934948,98.2378484 409.046275,98.1733959 409.175182,98.2554276 C409.304089,98.3374592 409.356823,98.2730068 409.333385,98.0620682 C409.567761,98.4605077 409.866586,98.7944887 410.229869,99.0640213 C410.593153,99.3335539 410.997445,99.5620673 411.44276,99.7495682 Z M416.259166,115.886287 C416.001353,116.823792 415.772839,117.532769 415.573619,118.01324 C415.3744,118.493711 415.075575,118.991753 414.677135,119.507381 C414.372446,119.460505 414.161511,119.495661 414.044323,119.612849 C413.927135,119.730038 413.79823,119.835505 413.657604,119.929256 C413.446665,119.976131 413.294323,119.976131 413.200573,119.929256 C413.106822,119.88238 413.024792,119.829647 412.954479,119.771053 C412.884166,119.712459 412.796276,119.653865 412.690807,119.595271 C412.585338,119.536677 412.427136,119.507381 412.216198,119.507381 C412.169322,119.31988 412.116589,119.1441 412.057994,118.980037 C411.9994,118.815974 411.841199,118.75738 411.583385,118.804256 C411.63026,118.616755 411.641979,118.458553 411.618541,118.329646 C411.595104,118.200739 411.565807,118.083553 411.530651,117.978084 C411.495494,117.872615 411.466198,117.767147 411.44276,117.661678 C411.419323,117.556208 411.431041,117.421444 411.477916,117.257381 C411.360728,117.257381 411.266979,117.17535 411.196666,117.011287 C411.126354,116.847224 411.044323,116.683163 410.950573,116.519099 C411.020885,116.097222 411.091197,115.698789 411.16151,115.323787 C411.231823,114.948785 411.319713,114.597226 411.425182,114.269099 C411.530651,113.940973 411.618541,113.612851 411.688854,113.284724 C411.759167,112.956598 411.724011,112.616757 411.583385,112.265193 C411.677136,112.335506 411.747447,112.317928 411.794323,112.212459 C411.841198,112.10699 411.876354,111.966366 411.899791,111.790584 C411.923229,111.614802 411.934948,111.439022 411.934948,111.26324 C411.934948,111.087458 411.923229,110.940975 411.899791,110.823787 C412.134168,110.214409 412.327525,109.587462 412.479869,108.942928 C412.632214,108.298393 412.772838,107.659728 412.901744,107.026912 C413.030651,106.394096 413.165416,105.773009 413.306041,105.163631 C413.446667,104.554253 413.634165,103.991758 413.868541,103.476131 C414.126355,103.429255 414.413462,103.458552 414.729869,103.564021 C415.046277,103.669491 415.239635,103.804255 415.309948,103.968318 C415.919326,103.499566 416.458383,102.931212 416.927135,102.26324 C417.395887,101.595268 417.876351,100.944884 418.368541,100.312068 C418.860731,99.6792525 419.405648,99.1167582 420.003307,98.6245682 C420.600966,98.1323782 421.356818,97.8394124 422.270885,97.745662 C422.739637,98.0972262 423.261117,98.4546445 423.835338,98.8179276 C424.40956,99.1812106 424.884164,99.6558153 425.259166,100.241756 C425.446667,100.335506 425.604869,100.435115 425.733776,100.540584 C425.862683,100.646053 426.020884,100.75738 426.208385,100.874568 L426.208385,101.296443 C426.489637,101.554257 426.71815,101.853082 426.893932,102.192928 C427.069714,102.532773 427.227916,102.896051 427.368541,103.282771 L427.790416,104.442928 C427.931042,104.829648 428.095103,105.198785 428.282604,105.550349 C428.235729,105.667538 428.212291,105.767146 428.212291,105.849178 C428.212291,105.931209 428.229869,106.019099 428.265026,106.112849 C428.300182,106.2066 428.329479,106.300349 428.352916,106.394099 C428.376354,106.48785 428.376354,106.616755 428.352916,106.780818 C428.282604,107.812073 427.948622,108.644096 427.350963,109.276912 C426.753304,109.909728 426.032608,110.431207 425.188854,110.841365 C424.3451,111.251523 423.442765,111.597223 422.481823,111.878474 C421.52088,112.159726 420.653701,112.46441 419.88026,112.792537 C419.88026,112.886287 419.921275,112.933162 420.003307,112.933162 C420.085339,112.933162 420.114635,112.991755 420.091198,113.108943 C420.6537,113.460507 421.099008,113.741755 421.427135,113.952693 C421.755262,114.163632 422.065806,114.339411 422.358776,114.480037 C422.651746,114.620663 422.968149,114.743708 423.307994,114.849178 C423.64784,114.954647 424.122445,115.077692 424.731823,115.218318 C425.341201,115.358944 425.815805,115.470271 426.155651,115.552303 C426.495496,115.634334 426.80604,115.704646 427.087291,115.76324 C427.368543,115.821834 427.655649,115.868709 427.948619,115.903865 C428.24159,115.939022 428.657601,115.980037 429.196666,116.026912 C429.407605,116.237851 429.589244,116.489801 429.741588,116.782771 C429.893933,117.075742 429.993541,117.368707 430.040416,117.661678 C430.087292,117.954648 430.09901,118.235895 430.075573,118.505428 C430.052135,118.77496 429.970105,118.991755 429.829479,119.155818 C429.852916,119.226131 429.90565,119.255428 429.987682,119.243709 C430.069714,119.23199 430.157603,119.237849 430.251354,119.261287 C430.110728,119.355037 429.993542,119.472224 429.899791,119.612849 L429.618541,120.034724 C429.524791,120.17535 429.419323,120.298396 429.302135,120.403865 C429.184947,120.509334 429.020886,120.573787 428.809948,120.597224 C428.083381,120.878476 427.169328,120.896054 426.06776,120.649959 C424.966192,120.403864 423.571675,119.952696 421.884166,119.296443 C421.040412,118.968317 420.354872,118.68121 419.827526,118.435115 C419.300179,118.18902 418.843153,117.942929 418.456432,117.696834 C418.069711,117.450739 417.718152,117.18707 417.401744,116.905818 C417.085337,116.624567 416.704481,116.284726 416.259166,115.886287 Z M421.28651,103.827693 C421.028696,104.202695 420.74159,104.542535 420.425182,104.847224 C420.108774,105.151913 419.80409,105.462457 419.511119,105.778865 C419.218149,106.095273 418.95448,106.446832 418.720104,106.833553 C418.485728,107.220273 418.321667,107.671441 418.227916,108.187068 C419.048233,108.093318 419.903693,107.864804 420.794323,107.501521 C421.684952,107.138238 422.528694,106.792539 423.325573,106.464412 C423.231822,105.737846 422.979872,105.151914 422.569713,104.706599 C422.159555,104.261285 421.731825,103.968319 421.28651,103.827693 Z M430.638073,103.722224 C430.802136,103.230034 431.065805,102.808164 431.429088,102.456599 C431.792371,102.105035 432.190805,101.80621 432.624401,101.560115 C433.057997,101.31402 433.509164,101.108944 433.977916,100.944881 C434.446669,100.780817 434.891977,100.640194 435.313854,100.523006 C436.251359,100.124566 437.253302,99.7437106 438.319713,99.3804276 C439.386125,99.0171445 440.475958,98.6948821 441.589244,98.4136307 C442.702531,98.1323793 443.845098,97.9507405 445.016979,97.8687088 C446.18886,97.7866772 447.419316,97.9331601 448.708385,98.308162 C448.778698,98.5425381 448.884166,98.7651921 449.024791,98.9761307 C449.165417,99.1870693 449.3119,99.3980046 449.464244,99.6089432 C449.616589,99.8198818 449.751353,100.024958 449.868541,100.224178 C449.985729,100.423397 450.056041,100.628473 450.079479,100.839412 C450.313855,101.050351 450.501353,101.278864 450.641979,101.524959 C450.782605,101.771054 450.806042,102.093316 450.712291,102.491756 C450.477915,102.679257 450.249402,102.849177 450.026744,103.001521 C449.804087,103.153866 449.563855,103.28863 449.306041,103.405818 C448.43885,103.19488 447.559952,103.124568 446.669323,103.194881 C445.778693,103.265194 444.981826,103.323787 444.278698,103.370662 C444.067759,103.159723 443.815808,103.054256 443.522838,103.054256 C443.229868,103.054256 442.925184,103.10113 442.608776,103.194881 L441.659557,103.476131 C441.343149,103.569881 441.044324,103.593318 440.763073,103.546443 C440.528696,103.733944 440.24159,103.85699 439.901744,103.915584 C439.561899,103.974178 439.192762,104.026912 438.794323,104.073787 C438.770885,104.190975 438.794322,104.267146 438.864635,104.302303 C438.934948,104.337459 438.946667,104.41363 438.899791,104.530818 C439.040417,104.694882 439.222056,104.812068 439.444713,104.882381 C439.667371,104.952694 439.901743,105.023005 440.147838,105.093318 C440.393933,105.163631 440.610728,105.251521 440.798229,105.35699 C440.98573,105.462459 441.114635,105.63238 441.184948,105.866756 C441.536512,105.866756 441.841196,105.948786 442.09901,106.112849 C442.356824,106.276913 442.567759,106.47613 442.731823,106.710506 C442.895886,106.944882 443.018932,107.196832 443.100963,107.466365 C443.182995,107.735898 443.22401,107.97613 443.22401,108.187068 C443.294323,108.374569 443.376353,108.4566 443.470104,108.433162 C443.563854,108.409724 443.669322,108.456599 443.78651,108.573787 C443.856823,108.878476 444.032602,109.18902 444.313854,109.505428 C444.595105,109.821835 444.817759,110.167535 444.981823,110.542537 C445.028698,111.128477 445.063854,111.649957 445.087291,112.10699 C445.110729,112.564024 445.028699,113.02691 444.841198,113.495662 C444.489633,114.362854 444.00331,115.136284 443.382213,115.815974 C442.761116,116.495665 442.057998,117.099175 441.272838,117.626521 C440.487678,118.153868 439.649796,118.605035 438.759166,118.980037 C437.868537,119.355039 437.001358,119.671442 436.157604,119.929256 C436.063853,119.905818 435.970104,119.905818 435.876354,119.929256 C435.782603,119.952693 435.759166,119.8941 435.806041,119.753474 C435.61854,119.917538 435.413464,120.023005 435.190807,120.069881 C434.96815,120.116756 434.727918,120.134334 434.470104,120.122615 C434.21229,120.110896 433.948621,120.093318 433.679088,120.069881 C433.409556,120.046443 433.157605,120.058162 432.923229,120.105037 C432.759166,119.940974 432.612683,119.782772 432.483776,119.630428 C432.354869,119.478083 432.25526,119.284726 432.184948,119.050349 C432.06776,119.050349 432.015026,118.991756 432.026744,118.874568 C432.038463,118.75738 431.997448,118.698787 431.903698,118.698787 C432.044323,118.464411 432.108776,118.329647 432.097057,118.29449 C432.085338,118.259334 432.050182,118.230037 431.991588,118.206599 C431.932994,118.183162 431.88026,118.153865 431.833385,118.118709 C431.78651,118.083552 431.774791,118.030819 431.798229,117.960506 C431.891979,117.773005 431.927135,117.55621 431.903698,117.310115 C431.88026,117.06402 431.856823,116.812069 431.833385,116.554256 C431.809948,116.296442 431.827525,116.050351 431.886119,115.815974 C431.944714,115.581598 432.114634,115.3941 432.395885,115.253474 C433.192764,115.347225 433.948616,115.317928 434.663463,115.165584 C435.378311,115.013239 436.052132,114.796445 436.684948,114.515193 C437.317763,114.233942 437.921273,113.929257 438.495494,113.601131 C439.069716,113.273004 439.63807,112.980038 440.200573,112.722224 C440.270885,112.511286 440.370494,112.335506 440.499401,112.194881 C440.628308,112.054255 440.763072,111.92535 440.903698,111.808162 C441.044323,111.690974 441.173228,111.562069 441.290416,111.421443 C441.407604,111.280817 441.489635,111.105038 441.53651,110.894099 C441.559948,110.730036 441.518933,110.630428 441.413463,110.595271 C441.307994,110.560115 441.25526,110.472225 441.25526,110.331599 L440.868541,110.226131 C440.423227,109.968317 439.919325,109.751522 439.356823,109.57574 C438.79432,109.399958 438.313856,109.206601 437.915416,108.995662 C437.657603,109.136288 437.417371,109.171444 437.194713,109.101131 C436.972056,109.030818 436.684949,108.960506 436.333385,108.890193 C435.536506,108.843318 434.768936,108.761288 434.030651,108.644099 C433.292366,108.526911 432.583389,108.315976 431.903698,108.011287 C431.856822,107.894099 431.798229,107.788631 431.727916,107.694881 C431.657604,107.60113 431.528699,107.577693 431.341198,107.624568 C431.200572,107.437067 431.059948,107.255428 430.919323,107.079646 C430.778697,106.903864 430.649792,106.710507 430.532604,106.499568 C430.579479,106.218317 430.614635,105.978085 430.638073,105.778865 C430.66151,105.579645 430.591198,105.386288 430.427135,105.198787 C430.520886,104.894098 430.591197,104.648007 430.638073,104.460506 C430.684948,104.273005 430.684948,104.026913 430.638073,103.722224 Z" id="Pointers" fill="#121212" mask="url(#mask-2)" transform="translate(374.539245, 106.998787) rotate(-14.000000) translate(-374.539245, -106.998787) "></path>
            </g>
        </g>
    </svg>
    """

public let spotifyLogoSvg =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="93px" height="28px" viewBox="0 0 93 28" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Desktop" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Pricing-/-Large" transform="translate(-382.000000, -3274.000000)">
                <g id="Group-8" transform="translate(348.000000, 3258.000000)">
                    <rect id="Rectangle" fill-opacity="0" fill="#FFFFFF" x="0" y="0" width="160" height="60"></rect>
                    <g id="Spotify_Logo_RGB_Black" transform="translate(34.000000, 16.000000)" fill="#7D7D7D">
                        <path d="M22.2155206,12.4128333 C17.7245689,9.741 10.3168444,9.49533333 6.0296941,10.7988333 C5.34126118,11.0081667 4.61323256,10.6186667 4.40460644,9.929 C4.19598032,9.239 4.58428444,8.51016667 5.27321646,8.3005 C10.1945635,6.804 18.3755689,7.093 23.5456369,10.1675 C24.1650268,10.5358333 24.3681628,11.337 24.0009875,11.9563333 C23.6336458,12.5766667 22.8332469,12.7811667 22.2155206,12.4128333 Z M22.0684508,16.3701667 C21.7535152,16.8823333 21.0850465,17.0428333 20.5744615,16.7285 C16.8303381,14.4228333 11.1210698,13.755 6.69150805,15.102 C6.11703757,15.2758333 5.51029159,14.9513333 5.33593739,14.377 C5.16274776,13.8015 5.48683363,13.1948333 6.06013953,13.0198333 C11.1204043,11.4816667 17.4107979,12.2266667 21.7110912,14.8741667 C22.2216762,15.1891667 22.3823882,15.8591667 22.0684508,16.3701667 Z M20.3636726,20.1706667 C20.1134544,20.5818333 19.578746,20.7108333 19.1698122,20.4601667 C15.8980089,18.457 11.7800555,18.0045 6.93041324,19.1143333 C6.46308408,19.2216667 5.9974186,18.9281667 5.89077639,18.4601667 C5.78363506,17.992 6.07544544,17.5255 6.54377281,17.4186667 C11.8509284,16.2031667 16.4034365,16.7263333 20.0756887,18.9743333 C20.4851216,19.2248333 20.6140572,19.7606667 20.3636726,20.1706667 Z M13.9742898,0.0395 C6.27974597,0.0395 0.0419248658,6.28816667 0.0419248658,13.9965 C0.0419248658,21.7053333 6.27974597,27.9538333 13.9742898,27.9538333 C21.669,27.9538333 27.9064884,21.7053333 27.9064884,13.9965 C27.9064884,6.28816667 21.669,0.0395 13.9742898,0.0395 Z" id="Fill-1"></path>
                        <path d="M37.9468283,12.9243333 C35.5413059,12.3496667 35.112907,11.9463333 35.112907,11.099 C35.112907,10.2981667 35.8655581,9.7595 36.9845528,9.7595 C38.0692755,9.7595 39.1446816,10.1686667 40.2724937,11.011 C40.3065993,11.0365 40.349356,11.0466667 40.3914472,11.04 C40.4335385,11.0336667 40.4706386,11.0106667 40.4954275,10.9756667 L41.6699893,9.317 C41.7182361,9.24866667 41.705093,9.15466667 41.6400429,9.10233333 C40.2979481,8.0235 38.7866565,7.499 37.0201556,7.499 C34.4228104,7.499 32.6085617,9.0605 32.6085617,11.2946667 C32.6085617,13.6905 34.1735903,14.5388333 36.8780769,15.1936667 C39.1799517,15.7248333 39.5684222,16.1696667 39.5684222,16.9653333 C39.5684222,17.8468333 38.7828301,18.3948333 37.5185957,18.3948333 C36.1146118,18.3948333 34.9691646,17.921 33.6879606,16.8095 C33.6561843,16.7821667 33.6125957,16.7695 33.5723345,16.7718333 C33.5300769,16.7753333 33.4913131,16.795 33.4643614,16.8275 L32.1473882,18.3976667 C32.0921538,18.4628333 32.0991413,18.5601667 32.1628605,18.6165 C33.6536887,19.9496667 35.4869034,20.654 37.4653578,20.654 C40.2641753,20.654 42.0727674,19.1218333 42.0727674,16.7506667 C42.0727674,14.7466667 40.877576,13.6383333 37.9468283,12.9243333" id="Fill-2"></path>
                        <path d="M50.4366118,15.6101667 C50.4366118,17.3023333 49.3961431,18.4831667 47.9063131,18.4831667 C46.4334526,18.4831667 45.3224436,17.2486667 45.3224436,15.6101667 C45.3224436,13.9718333 46.4334526,12.7371667 47.9063131,12.7371667 C49.372186,12.7371667 50.4366118,13.9451667 50.4366118,15.6101667 Z M48.4049195,10.5476667 C47.1917603,10.5476667 46.1967102,11.0263333 45.3761807,12.0071667 L45.3761807,10.9031667 C45.3761807,10.816 45.3056404,10.745 45.2186297,10.745 L43.0648229,10.745 C42.9778122,10.745 42.9074383,10.816 42.9074383,10.9031667 L42.9074383,23.1701667 C42.9074383,23.2573333 42.9778122,23.3283333 43.0648229,23.3283333 L45.2186297,23.3283333 C45.3056404,23.3283333 45.3761807,23.2573333 45.3761807,23.1701667 L45.3761807,19.2981667 C46.1968766,20.2208333 47.192093,20.6715 48.4049195,20.6715 C50.6590465,20.6715 52.9409571,18.9331667 52.9409571,15.6101667 C52.9409571,12.2865 50.6590465,10.5476667 48.4049195,10.5476667 Z" id="Fill-3"></path>
                        <path d="M58.7901413,18.5008333 C57.2460751,18.5008333 56.0819946,17.258 56.0819946,15.6101667 C56.0819946,13.9555 57.205814,12.7545 58.7545385,12.7545 C60.3085868,12.7545 61.4804866,13.9973333 61.4804866,15.6463333 C61.4804866,17.3008333 60.3490143,18.5008333 58.7901413,18.5008333 Z M58.7901413,10.5476667 C55.8871771,10.5476667 53.6129195,12.787 53.6129195,15.6463333 C53.6129195,18.4745 55.8715385,20.6903333 58.7545385,20.6903333 C61.6678175,20.6903333 63.9490626,18.4585 63.9490626,15.6101667 C63.9490626,12.7713333 61.6834562,10.5476667 58.7901413,10.5476667 Z" id="Fill-4"></path>
                        <path d="M70.1482862,10.745 L67.778034,10.745 L67.778034,8.31733333 C67.778034,8.23016667 67.7078265,8.15933333 67.6208157,8.15933333 L65.4671753,8.15933333 C65.3799982,8.15933333 65.3091252,8.23016667 65.3091252,8.31733333 L65.3091252,10.745 L64.2734812,10.745 C64.1866369,10.745 64.1165957,10.816 64.1165957,10.9031667 L64.1165957,12.7576667 C64.1165957,12.8446667 64.1866369,12.9156667 64.2734812,12.9156667 L65.3091252,12.9156667 L65.3091252,17.7143333 C65.3091252,19.6535 66.2725653,20.6366667 68.1726601,20.6366667 C68.9452755,20.6366667 69.5862934,20.4768333 70.1903775,20.1336667 C70.2394562,20.1061667 70.270068,20.0531667 70.270068,19.9968333 L70.270068,18.2308333 C70.270068,18.1763333 70.241619,18.125 70.1952021,18.0963333 C70.1482862,18.0668333 70.0898909,18.0653333 70.0418104,18.0891667 C69.6268873,18.2983333 69.2257728,18.3948333 68.7774097,18.3948333 C68.0864812,18.3948333 67.778034,18.0806667 67.778034,17.3761667 L67.778034,12.9156667 L70.1482862,12.9156667 C70.235297,12.9156667 70.3053381,12.8446667 70.3053381,12.7576667 L70.3053381,10.9031667 C70.3053381,10.816 70.235297,10.745 70.1482862,10.745" id="Fill-5"></path>
                        <path d="M78.4063202,10.7545 L78.4063202,10.4563333 C78.4063202,9.57916667 78.7420519,9.188 79.4950358,9.188 C79.9440644,9.188 80.3047513,9.27733333 80.7086941,9.41233333 C80.7584383,9.428 80.8103453,9.42016667 80.8507728,9.39033333 C80.8923649,9.3605 80.9158229,9.31266667 80.9158229,9.262 L80.9158229,7.44366667 C80.9158229,7.37433333 80.8712361,7.31266667 80.8045224,7.29216667 C80.3777871,7.16516667 79.8317657,7.0345 79.0142308,7.0345 C77.0246297,7.0345 75.9731807,8.15683333 75.9731807,10.279 L75.9731807,10.7356667 L74.9383685,10.7356667 C74.8515242,10.7356667 74.7803184,10.8066667 74.7803184,10.8936667 L74.7803184,12.7576667 C74.7803184,12.8446667 74.8515242,12.9156667 74.9383685,12.9156667 L75.9731807,12.9156667 L75.9731807,20.3173333 C75.9731807,20.4045 76.0433882,20.4753333 76.1302326,20.4753333 L78.2840394,20.4753333 C78.3710501,20.4753333 78.4419231,20.4045 78.4419231,20.3173333 L78.4419231,12.9156667 L80.4529857,12.9156667 L83.5336315,20.3153333 C83.1839249,21.0928333 82.8400411,21.2475 82.3705492,21.2475 C81.9910626,21.2475 81.5914454,21.134 81.1828444,20.91 C81.1444132,20.8888333 81.0989946,20.8853333 81.0574025,20.8981667 C81.0163095,20.9128333 80.9817048,20.9433333 80.9642361,20.9835 L80.2342111,22.588 C80.1994401,22.6638333 80.2290537,22.7528333 80.3019231,22.7921667 C81.0640572,23.2056667 81.7521574,23.3821667 82.6023005,23.3821667 C84.1927835,23.3821667 85.0718748,22.6398333 85.8469857,20.6431667 L89.5837889,10.9698333 C89.6024222,10.9211667 89.596932,10.8663333 89.5671521,10.823 C89.5375385,10.7803333 89.4897907,10.7545 89.437551,10.7545 L87.1952361,10.7545 C87.1281896,10.7545 87.0679642,10.7971667 87.0460036,10.8605 L84.7489535,17.4333333 L82.2329624,10.8561667 C82.2100036,10.7948333 82.1512755,10.7545 82.0858927,10.7545 L78.4063202,10.7545" id="Fill-6"></path>
                        <path d="M73.6188998,10.745 L71.4649267,10.745 C71.3779159,10.745 71.3072093,10.816 71.3072093,10.9031667 L71.3072093,20.3173333 C71.3072093,20.4045 71.3779159,20.4753333 71.4649267,20.4753333 L73.6188998,20.4753333 C73.7057442,20.4753333 73.7766172,20.4045 73.7766172,20.3173333 L73.7766172,10.9031667 C73.7766172,10.816 73.7057442,10.745 73.6188998,10.745" id="Fill-7"></path>
                        <path d="M72.5528104,6.4585 C71.6996726,6.4585 71.0070805,7.15066667 71.0070805,8.0055 C71.0070805,8.86066667 71.6996726,9.5535 72.5528104,9.5535 C73.4057818,9.5535 74.097542,8.86066667 74.097542,8.0055 C74.097542,7.15066667 73.4057818,6.4585 72.5528104,6.4585" id="Fill-8"></path>
                        <path d="M91.5108354,11.6815 L91.1162093,11.6815 L91.1162093,12.1856667 L91.5108354,12.1856667 C91.7078157,12.1856667 91.8254383,12.0891667 91.8254383,11.9333333 C91.8254383,11.7693333 91.7078157,11.6815 91.5108354,11.6815 Z M91.7667102,12.4006667 L92.1954419,13.002 L91.8339231,13.002 L91.4479481,12.4505 L91.1162093,12.4505 L91.1162093,13.002 L90.813585,13.002 L90.813585,11.4081667 L91.5231467,11.4081667 C91.8926512,11.4081667 92.1358819,11.5975 92.1358819,11.9165 C92.1358819,12.1778333 91.9851521,12.3375 91.7667102,12.4006667 Z M91.4306458,10.8776667 C90.6542039,10.8776667 90.0665903,11.496 90.0665903,12.2531667 C90.0665903,13.0098333 90.6500447,13.62 91.4226601,13.62 C92.1989356,13.62 92.7870483,13.002 92.7870483,12.2445 C92.7870483,11.4876667 92.2032612,10.8776667 91.4306458,10.8776667 Z M91.4226601,13.7718333 C90.5701878,13.7718333 89.9068766,13.086 89.9068766,12.2531667 C89.9068766,11.4201667 90.578839,10.7266667 91.4306458,10.7266667 C92.2829517,10.7266667 92.9464293,11.4123333 92.9464293,12.2445 C92.9464293,13.0773333 92.274966,13.7718333 91.4226601,13.7718333 Z" id="Fill-9"></path>
                    </g>
                </g>
            </g>
        </g>
    </svg>
    """

public let venmoLogoSvg =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="80px" height="16px" viewBox="0 0 80 16" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Desktop" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Pricing-/-Large" transform="translate(-537.000000, -3282.000000)" fill="#808080">
                <g id="venmo_logo_blue" transform="translate(537.000000, 3282.000000)">
                    <path d="M13.2008304,0.138269896 C13.7399308,1.02865052 13.9829758,1.94574394 13.9829758,3.10422145 C13.9829758,6.79916955 10.8287889,11.5991696 8.26878893,14.9696886 L2.4215917,14.9696886 L0.0765397924,0.947266436 L5.19640138,0.461176471 L6.43626298,10.4390311 C7.59474048,8.5516955 9.02435986,5.58574394 9.02435986,3.56359862 C9.02435986,2.4567474 8.83474048,1.70283737 8.5384083,1.08207612 L13.2008304,0.138269896 Z" id="Path"></path>
                    <path d="M19.8364014,6.31294118 C20.7785467,6.31294118 23.1504498,5.88193772 23.1504498,4.53384083 C23.1504498,3.88650519 22.6927336,3.56359862 22.1533564,3.56359862 C21.2096886,3.56359862 19.9713495,4.69522491 19.8364014,6.31294118 Z M19.7284429,8.98283737 C19.7284429,10.6289273 20.643737,11.2747405 21.8571626,11.2747405 C23.1785467,11.2747405 24.443737,10.9518339 26.0881661,10.1161246 L25.4687889,14.3220761 C24.310173,14.8881661 22.5044983,15.2658824 20.7518339,15.2658824 C16.3060208,15.2658824 14.7148789,12.5696886 14.7148789,9.19903114 C14.7148789,4.83031142 17.3031142,0.191557093 22.6391696,0.191557093 C25.5770242,0.191557093 27.2197924,1.83750865 27.2197924,4.12941176 C27.2200692,7.82422145 22.4779239,8.95612457 19.7284429,8.98283737 Z" id="Shape" fill-rule="nonzero"></path>
                    <path d="M41.9950173,3.42878893 C41.9950173,3.96802768 41.913218,4.75017301 41.8318339,5.26131488 L40.2955017,14.9695502 L35.310173,14.9695502 L36.7115571,6.07017301 C36.7381315,5.82878893 36.8197924,5.34283737 36.8197924,5.07321799 C36.8197924,4.42588235 36.4153633,4.26435986 35.9291349,4.26435986 C35.2833218,4.26435986 34.6359862,4.56069204 34.2048443,4.77702422 L32.6153633,14.9696886 L27.6019377,14.9696886 L29.8923183,0.434740484 L34.2314187,0.434740484 L34.2863668,1.59487889 C35.3100346,0.92083045 36.6579931,0.19183391 38.570519,0.19183391 C41.1043599,0.191557093 41.9950173,1.48636678 41.9950173,3.42878893 Z" id="Path"></path>
                    <path d="M56.7950173,1.78269896 C58.2228374,0.759031142 59.5710727,0.191557093 61.4300346,0.191557093 C63.9898962,0.191557093 64.8802768,1.48636678 64.8802768,3.42878893 C64.8802768,3.96802768 64.7987543,4.75017301 64.7173702,5.26131488 L63.1828374,14.9695502 L58.1961246,14.9695502 L59.6242215,5.88207612 C59.650519,5.63916955 59.7057439,5.34283737 59.7057439,5.15487889 C59.7057439,4.42602076 59.3011765,4.26435986 58.8150865,4.26435986 C58.1958478,4.26435986 57.5768858,4.53397924 57.1173702,4.77702422 L55.5281661,14.9696886 L50.542699,14.9696886 L51.9707958,5.88221453 C51.9970934,5.63930796 52.0506574,5.34297578 52.0506574,5.1550173 C52.0506574,4.42615917 51.6459516,4.26449827 51.1615225,4.26449827 C50.5143253,4.26449827 49.8683737,4.56083045 49.4372318,4.77716263 L47.8465052,14.969827 L42.8347405,14.969827 L45.1248443,0.434878893 L49.4109343,0.434878893 L49.5456055,1.6483045 C50.542699,0.921107266 51.8895502,0.192110727 53.6950865,0.192110727 C55.2584083,0.191557093 56.2819377,0.865743945 56.7950173,1.78269896 Z" id="Path"></path>
                    <path d="M74.8019377,6.01688581 C74.8019377,4.83031142 74.5051903,4.02131488 73.616609,4.02131488 C71.6491349,4.02131488 71.2451211,7.49965398 71.2451211,9.27903114 C71.2451211,10.6289273 71.6228374,11.4643599 72.5111419,11.4643599 C74.3707958,11.4643599 74.8019377,7.79612457 74.8019377,6.01688581 Z M66.1786851,9.06422145 C66.1786851,4.48055363 68.6035986,0.191557093 74.1828374,0.191557093 C78.3868512,0.191557093 79.9235986,2.67307958 79.9235986,6.0982699 C79.9235986,10.6287889 77.5248443,15.3205536 71.8110727,15.3205536 C67.5799308,15.3205536 66.1786851,12.542699 66.1786851,9.06422145 Z" id="Shape" fill-rule="nonzero"></path>
                </g>
            </g>
        </g>
    </svg>
    """

public let nytLogoSvg =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="140px" height="19px" viewBox="0 0 140 19" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Desktop" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Pricing-/-Large" transform="translate(-190.000000, -3279.000000)">
                <g id="Group-6" transform="translate(180.000000, 3258.000000)">
                    <rect id="Rectangle" fill-opacity="0" fill="#FFFFFF" x="0" y="0" width="160" height="60"></rect>
                    <g id="NewYorkTimes" transform="translate(10.000000, 21.000000)" fill="#7D7D7D" fill-rule="nonzero">
                        <path d="M10.4432432,2.204 C10.4432432,0.684 9.00540541,0.304 7.87027027,0.304 L7.87027027,0.532 C8.55135135,0.532 9.08108108,0.76 9.08108108,1.292 C9.08108108,1.596 8.85405405,2.052 8.17297297,2.052 C7.64324324,2.052 6.50810811,1.748 5.67567568,1.444 C4.69189189,1.064 3.78378378,0.76 3.02702703,0.76 C1.51351351,0.76 0.454054054,1.9 0.454054054,3.192 C0.454054054,4.332 1.28648649,4.712 1.58918919,4.864 L1.66486486,4.712 C1.51351351,4.56 1.28648649,4.408 1.28648649,3.952 C1.28648649,3.648 1.58918919,3.116 2.34594595,3.116 C3.02702703,3.116 3.93513514,3.42 5.14594595,3.8 C6.20540541,4.104 7.34054054,4.332 7.94594595,4.408 L7.94594595,6.764 L6.81081081,7.752 L6.81081081,7.828 L7.94594595,8.816 L7.94594595,12.084 C7.34054054,12.464 6.65945946,12.54 6.05405405,12.54 C4.91891892,12.54 3.93513514,12.236 3.1027027,11.324 L6.20540541,9.804 L6.20540541,4.56 L2.42162162,6.232 C2.72432432,5.244 3.55675676,4.56 4.38918919,4.104 L4.31351351,3.876 C2.04324324,4.484 0,6.612 0,9.196 C0,12.236 2.4972973,14.516 5.2972973,14.516 C8.32432432,14.516 10.2918919,12.084 10.2918919,9.576 L10.1405405,9.576 C9.68648649,10.564 9.00540541,11.476 8.17297297,11.932 L8.17297297,8.816 L9.38378378,7.828 L9.38378378,7.752 L8.17297297,6.764 L8.17297297,4.408 C9.30810811,4.408 10.4432432,3.648 10.4432432,2.204 Z M3.85945946,10.564 L2.95135135,11.02 C2.42162162,10.336 2.11891892,9.424 2.11891892,8.132 C2.11891892,7.6 2.11891892,6.992 2.27027027,6.536 L3.85945946,5.852 L3.85945946,10.564 Z M11.8810811,12.312 L10.8972973,13.072 L11.0486486,13.224 L11.5027027,12.844 L13.1675676,14.364 L15.4378378,12.844 L15.3621622,12.692 L14.7567568,13.072 L14,12.312 L14,7.144 L14.6054054,6.688 L15.8918919,7.752 L15.8918919,12.388 C15.8918919,15.276 15.2864865,15.732 14,16.188 L14,16.416 C16.1189189,16.492 18.0864865,15.808 18.0864865,12.084 L18.0864865,7.068 L18.7675676,6.536 L18.6162162,6.384 L18.0108108,6.84 L16.1189189,5.244 L14,6.84 L14,0.608 L13.8486486,0.608 L11.2,2.432 L11.2,2.584 C11.5027027,2.736 11.9567568,2.888 11.9567568,3.724 L11.8810811,12.312 Z M25.7297297,11.476 L23.8378378,12.92 L21.9459459,11.4 L21.9459459,10.488 L25.5027027,8.056 L25.5027027,7.98 L23.6864865,5.244 L19.7513514,7.372 L19.7513514,12.388 L18.9945946,12.996 L19.1459459,13.148 L19.827027,12.616 L22.4,14.516 L25.8054054,11.78 L25.7297297,11.476 Z M21.9459459,10.184 L21.9459459,6.46 L22.0972973,6.384 L23.7621622,9.044 L21.9459459,10.184 Z M40.1837838,1.52 C40.1837838,1.292 40.1081081,1.064 40.0324324,0.836 L39.8810811,0.836 C39.6540541,1.444 39.3513514,1.748 38.5945946,1.748 C37.9135135,1.748 37.4594595,1.368 37.1567568,1.064 L34.9621622,3.572 L35.1135135,3.724 L35.8702703,3.04 C36.3243243,3.42 36.7027027,3.724 37.7621622,3.8 L37.7621622,10.108 L33.2972973,2.432 C32.9189189,1.824 32.3891892,0.988 31.3297297,0.988 C30.1189189,0.988 29.0594595,2.052 29.2108108,3.724 L29.4378378,3.724 C29.5135135,3.268 29.7405405,2.736 30.2702703,2.736 C30.6486486,2.736 31.027027,3.116 31.2540541,3.496 L31.2540541,6.004 C29.8918919,6.004 28.9837838,6.612 28.9837838,7.752 C28.9837838,8.36 29.2864865,9.272 30.1945946,9.5 L30.1945946,9.348 C30.0432432,9.196 29.9675676,9.044 29.9675676,8.816 C29.9675676,8.436 30.2702703,8.132 30.8,8.132 L31.1783784,8.132 L31.1783784,11.324 C29.5891892,11.324 28.3027027,12.236 28.3027027,13.756 C28.3027027,15.2 29.5135135,15.884 30.8756757,15.808 L30.8756757,15.656 C30.0432432,15.58 29.6648649,15.2 29.6648649,14.668 C29.6648649,13.984 30.1189189,13.68 30.7243243,13.68 C31.3297297,13.68 31.8594595,14.06 32.2378378,14.516 L34.4324324,12.084 L34.2810811,11.932 L33.7513514,12.54 C32.9189189,11.78 32.4648649,11.552 31.4810811,11.4 L31.4810811,3.8 L37.5351351,14.44 L37.9891892,14.44 L37.9891892,3.8 C39.1243243,3.724 40.1837838,2.812 40.1837838,1.52 Z M45.7081081,11.476 L43.8162162,12.92 L41.9243243,11.4 L41.9243243,10.488 L45.4810811,8.056 L45.4810811,7.98 L43.6648649,5.244 L39.7297297,7.372 L39.7297297,12.388 L38.972973,12.996 L39.1243243,13.148 L39.8054054,12.616 L42.3783784,14.516 L45.7837838,11.78 L45.7081081,11.476 Z M41.9243243,10.184 L41.9243243,6.46 L42.0756757,6.384 L43.7405405,9.044 L41.9243243,10.184 Z M58.0432432,6.08 L57.5135135,6.46 L56.0756757,5.244 L54.4108108,6.764 L55.0918919,7.448 L55.0918919,13.148 L53.2756757,12.008 L53.2756757,7.296 L53.8810811,6.916 L52.1405405,5.244 L50.4756757,6.764 L51.1567568,7.448 L51.1567568,12.92 L50.9297297,13.072 L49.3405405,11.932 L49.3405405,7.372 C49.3405405,6.308 48.8108108,6.004 48.2054054,5.624 C47.6756757,5.244 47.372973,5.016 47.372973,4.484 C47.372973,4.028 47.827027,3.8 48.0540541,3.648 L48.0540541,3.496 C47.4486486,3.496 45.8594595,4.104 45.8594595,5.548 C45.8594595,6.308 46.2378378,6.612 46.6162162,6.992 C46.9945946,7.372 47.372973,7.676 47.372973,8.36 L47.372973,12.768 L46.5405405,13.376 L46.6918919,13.528 L47.4486486,12.92 L49.1891892,14.44 L51.0810811,13.148 L53.2,14.44 L57.2108108,12.084 L57.2108108,6.992 L58.1945946,6.232 L58.0432432,6.08 Z M72.1189189,1.9 L71.3621622,2.584 L69.6972973,1.064 L67.2,2.888 L67.2,1.216 L66.972973,1.216 L67.0486486,13.528 C66.8216216,13.528 66.1405405,13.376 65.6108108,13.224 L65.4594595,2.964 C65.4594595,2.204 64.9297297,1.14 63.5675676,1.14 C62.2054054,1.14 61.2972973,2.204 61.2972973,3.268 L61.5243243,3.268 C61.6,2.812 61.827027,2.432 62.2810811,2.432 C62.7351351,2.432 63.1135135,2.736 63.1135135,3.724 L63.1135135,6.688 C61.7513514,6.764 60.9189189,7.524 60.9189189,8.512 C60.9189189,9.12 61.2216216,10.032 62.1297297,10.032 L62.1297297,9.88 C61.827027,9.728 61.7513514,9.5 61.7513514,9.348 C61.7513514,8.892 62.1297297,8.74 62.7351351,8.74 L63.0378378,8.74 L63.0378378,13.452 C61.9027027,13.832 61.4486486,14.668 61.4486486,15.58 C61.4486486,16.872 62.4324324,17.784 63.9459459,17.784 C65.0054054,17.784 65.9135135,17.632 66.8216216,17.404 C67.5783784,17.252 68.5621622,17.024 69.0162162,17.024 C69.6216216,17.024 69.8486486,17.328 69.8486486,17.708 C69.8486486,18.24 69.6216216,18.468 69.3189189,18.544 L69.3189189,18.696 C70.5297297,18.468 71.2864865,17.708 71.2864865,16.568 C71.2864865,15.428 70.1513514,14.744 68.9405405,14.744 C68.3351351,14.744 67.0486486,14.972 66.1405405,15.124 C65.0810811,15.352 64.0216216,15.504 63.7189189,15.504 C63.1891892,15.504 62.5837838,15.276 62.5837838,14.516 C62.5837838,13.908 63.1135135,13.376 64.4,13.376 C65.0810811,13.376 65.9135135,13.452 66.7459459,13.68 C67.6540541,13.908 68.4864865,14.136 69.2432432,14.136 C70.3783784,14.136 71.3621622,13.756 71.3621622,12.16 L71.3621622,2.812 L72.2702703,2.052 L72.1189189,1.9 Z M69.0162162,6.536 C68.7891892,6.764 68.4864865,6.992 68.1081081,6.992 C67.7297297,6.992 67.3513514,6.764 67.2,6.536 L67.2,3.192 L67.9567568,2.66 L69.0162162,3.648 L69.0162162,6.536 Z M69.0162162,8.816 C68.8648649,8.664 68.4864865,8.436 68.1081081,8.436 C67.7297297,8.436 67.3513514,8.664 67.2,8.816 L67.2,6.84 C67.3513514,6.992 67.7297297,7.22 68.1081081,7.22 C68.4864865,7.22 68.8648649,6.992 69.0162162,6.84 L69.0162162,8.816 Z M69.0162162,12.388 C69.0162162,12.996 68.6378378,13.604 67.8054054,13.604 L67.2,13.604 L67.2,9.12 C67.3513514,8.968 67.7297297,8.74 68.1081081,8.74 C68.4864865,8.74 68.7891892,8.968 69.0162162,9.12 L69.0162162,12.388 Z M79.3837838,6.992 L76.9621622,5.244 L73.2540541,7.372 L73.2540541,12.312 L72.4972973,12.92 L72.572973,13.072 L73.1783784,12.616 L75.6,14.44 L79.3837838,12.16 L79.3837838,6.992 Z M75.2972973,11.78 L75.2972973,6.308 L77.1891892,7.676 L77.1891892,13.072 L75.2972973,11.78 Z M86.572973,5.396 L86.4216216,5.396 C86.1945946,5.548 85.9675676,5.7 85.7405405,5.7 C85.4378378,5.7 85.0594595,5.548 84.9081081,5.32 L84.7567568,5.32 L83.4702703,6.764 L82.1837838,5.32 L79.9135135,6.84 L79.9891892,6.992 L80.5945946,6.612 L81.3513514,7.448 L81.3513514,12.236 L80.3675676,12.996 L80.5189189,13.148 L80.972973,12.768 L82.7891892,14.288 L85.1351351,12.692 L85.0594595,12.54 L84.3783784,12.92 L83.4702703,12.16 L83.4702703,6.84 C83.8486486,7.22 84.3027027,7.6 84.8324324,7.6 C85.8918919,7.676 86.4972973,6.612 86.572973,5.396 Z M95.6540541,12.692 L93.0810811,14.44 L89.6,9.12 L92.0972973,5.244 L92.2486486,5.244 C92.5513514,5.548 93.0054054,5.852 93.5351351,5.852 C94.0648649,5.852 94.4432432,5.548 94.6702703,5.244 L94.8216216,5.244 C94.7459459,6.764 93.6864865,7.676 92.9297297,7.676 C92.172973,7.676 91.7945946,7.296 91.3405405,7.068 L91.1135135,7.448 L94.8972973,13.072 L95.6540541,12.616 L95.6540541,12.692 Z M87.3297297,12.312 L86.3459459,13.072 L86.4972973,13.224 L86.9513514,12.844 L88.6162162,14.364 L90.8864865,12.844 L90.7351351,12.692 L90.1297297,13.072 L89.372973,12.312 L89.372973,0.608 L89.2972973,0.608 L86.572973,2.432 L86.572973,2.584 C86.8756757,2.736 87.3297297,2.812 87.3297297,3.724 L87.3297297,12.312 Z M108.216216,2.204 C108.216216,0.684 106.778378,0.304 105.643243,0.304 L105.643243,0.532 C106.324324,0.532 106.854054,0.76 106.854054,1.292 C106.854054,1.596 106.627027,2.052 105.945946,2.052 C105.416216,2.052 104.281081,1.748 103.448649,1.444 C102.464865,1.14 101.556757,0.836 100.8,0.836 C99.2864865,0.836 98.227027,1.976 98.227027,3.268 C98.227027,4.408 99.0594595,4.788 99.3621622,4.94 L99.4378378,4.788 C99.2108108,4.636 98.9837838,4.484 98.9837838,4.028 C98.9837838,3.724 99.2864865,3.192 100.043243,3.192 C100.724324,3.192 101.632432,3.496 102.843243,3.876 C103.902703,4.18 105.037838,4.408 105.643243,4.484 L105.643243,6.84 L104.508108,7.828 L104.508108,7.904 L105.643243,8.892 L105.643243,12.16 C105.037838,12.54 104.356757,12.616 103.751351,12.616 C102.616216,12.616 101.632432,12.312 100.8,11.4 L103.902703,9.88 L103.902703,4.56 L100.118919,6.232 C100.497297,5.244 101.32973,4.56 102.086486,4.028 L102.010811,3.876 C99.7405405,4.484 97.6972973,6.536 97.6972973,9.12 C97.6972973,12.16 100.194595,14.44 102.994595,14.44 C106.021622,14.44 107.989189,12.008 107.989189,9.5 L107.837838,9.5 C107.383784,10.488 106.702703,11.4 105.87027,11.856 L105.87027,8.74 L107.081081,7.752 L107.081081,7.676 L105.945946,6.688 L105.945946,4.408 C107.081081,4.408 108.216216,3.648 108.216216,2.204 Z M101.632432,10.564 L100.724324,11.02 C100.194595,10.336 99.8918919,9.424 99.8918919,8.132 C99.8918919,7.6 99.9675676,6.992 100.118919,6.536 L101.708108,5.852 L101.632432,10.564 Z M110.864865,1.444 L110.789189,1.444 L109.275676,2.736 L109.275676,2.812 L110.562162,4.256 L110.713514,4.256 L112.227027,2.964 L112.227027,2.888 L110.864865,1.444 Z M113.135135,12.692 L112.52973,13.072 L111.772973,12.312 L111.772973,7.068 L112.52973,6.536 L112.378378,6.384 L111.848649,6.84 L110.486486,5.244 L108.291892,6.764 L108.443243,6.992 L108.972973,6.612 L109.654054,7.448 L109.654054,12.388 L108.67027,13.148 L108.745946,13.3 L109.275676,12.92 L110.940541,14.44 L113.210811,12.92 L113.135135,12.692 Z M125.772973,12.616 L125.243243,12.996 L124.410811,12.236 L124.410811,7.068 L125.167568,6.46 L125.016216,6.308 L124.410811,6.84 L122.67027,5.244 L120.4,6.84 L118.659459,5.244 L116.540541,6.84 L115.178378,5.244 L112.983784,6.764 L113.059459,6.992 L113.589189,6.612 L114.345946,7.448 L114.345946,12.388 L113.740541,12.996 L115.481081,14.44 L117.145946,12.92 L116.464865,12.236 L116.464865,7.068 L117.145946,6.612 L118.281081,7.676 L118.281081,12.236 L117.675676,12.844 L119.416216,14.288 L121.081081,12.768 L120.4,12.084 L120.4,7.068 L121.005405,6.688 L122.216216,7.752 L122.216216,12.312 L121.686486,12.844 L123.427027,14.44 L125.772973,12.844 L125.772973,12.616 Z M132.356757,11.476 L130.464865,12.92 L128.572973,11.4 L128.572973,10.488 L132.12973,8.056 L132.12973,7.98 L130.313514,5.244 L126.378378,7.372 L126.378378,12.54 L129.027027,14.44 L132.432432,11.704 L132.356757,11.476 Z M128.572973,10.184 L128.572973,6.46 L128.724324,6.384 L130.389189,9.044 L128.572973,10.184 Z M139.243243,9.5 L137.805405,8.36 C138.789189,7.524 139.167568,6.384 139.167568,5.624 L139.167568,5.168 L139.016216,5.168 C138.864865,5.548 138.562162,5.928 137.956757,5.928 C137.351351,5.928 136.972973,5.624 136.594595,5.168 L133.189189,7.068 L133.189189,9.804 L134.475676,10.792 C133.189189,11.932 132.962162,12.692 132.962162,13.3 C132.962162,14.06 133.340541,14.592 133.945946,14.82 L134.021622,14.668 C133.87027,14.516 133.718919,14.44 133.718919,14.06 C133.718919,13.832 134.021622,13.452 134.627027,13.452 C135.383784,13.452 135.837838,13.984 136.064865,14.212 L139.318919,12.236 L139.318919,9.5 L139.243243,9.5 Z M138.410811,7.22 C137.881081,8.132 136.745946,9.044 136.064865,9.5 L135.232432,8.816 L135.232432,6.156 C135.535135,6.916 136.367568,7.524 137.2,7.524 C137.72973,7.524 138.032432,7.448 138.410811,7.22 Z M137.124324,13.3 C136.745946,12.464 135.837838,11.856 134.92973,11.856 C134.702703,11.856 134.097297,11.856 133.491892,12.236 C133.87027,11.628 134.854054,10.564 136.140541,9.804 L137.048649,10.564 L137.124324,13.3 Z" id="Shape"></path>
                    </g>
                </g>
            </g>
        </g>
    </svg>
    """

public let atlassianLogoSvg =
base64EncodedString <| """
    <?xml version="1.0" encoding="UTF-8"?>
    <svg width="112px" height="14px" viewBox="0 0 112 14" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
        <g id="Desktop" stroke="none" stroke-width="1" fill="none" fill-rule="evenodd">
            <g id="Pricing-/-Large" transform="translate(-708.000000, -3281.000000)">
                <g id="Group-5" transform="translate(684.000000, 3258.000000)">
                    <rect id="Rectangle" fill-opacity="0" fill="#FFFFFF" x="0" y="0" width="160" height="60"></rect>
                    <g id="Atlassian-horizontal-neutral-onecolor-rgb" transform="translate(24.000000, 23.000000)" fill="#7D7D7D" fill-rule="nonzero">
                        <path d="M4.07348148,6.49764706 C3.99059613,6.38986355 3.85693372,6.33299426 3.72114189,6.34773728 C3.58535005,6.36248031 3.46723197,6.4466856 3.40977778,6.56970588 L0.0435555556,13.2505882 C-0.0191274839,13.3746006 -0.0124853726,13.5220191 0.0611001242,13.6399816 C0.134685621,13.7579442 0.264670646,13.8295483 0.404444444,13.8291176 L5.09185185,13.8291176 C5.24549403,13.8331524 5.38704223,13.7467484 5.45274074,13.6088235 C6.46074074,11.5294118 5.84888889,8.37941176 4.07348148,6.49764706 Z" id="Path"></path>
                        <path d="M6.5437037,0.411764706 C4.83083405,3.01842547 4.62980946,6.32854931 6.01481481,9.12058824 L8.28592593,13.6088235 C8.35438414,13.7442651 8.49415996,13.8295872 8.64681481,13.8291176 L13.3342222,13.8291176 C13.4728875,13.8274211 13.6009002,13.7549644 13.6729909,13.6373706 C13.7450816,13.5197769 13.7510891,13.3736209 13.6888889,13.2505882 L7.22814815,0.411764706 C7.16500067,0.28108477 7.03193545,0.197946485 6.88592593,0.197946485 C6.7399164,0.197946485 6.60685118,0.28108477 6.5437037,0.411764706 Z" id="Path"></path>
                        <path d="M61.5253333,5.77705882 C61.5253333,7.42411765 62.298963,8.74588235 65.322963,9.32441176 C67.1274074,9.69911765 67.5048889,9.98941176 67.5048889,10.5864706 C67.5048889,11.1835294 67.1274074,11.5417647 65.8456296,11.5417647 C64.3092731,11.5142162 62.8036692,11.1096383 61.4631111,10.3641176 L61.4631111,13.0611765 C62.3736296,13.5038235 63.5765926,14 65.8186667,14 C68.9816296,14 70.2364444,12.6020588 70.2364444,10.5205882 M70.2364444,10.5205882 C70.2364444,8.55852941 69.189037,7.63823529 66.2314074,7.00617647 C64.5991111,6.64794118 64.202963,6.28970588 64.202963,5.77088235 C64.202963,5.12235294 64.7878519,4.85058824 65.8622222,4.85058824 C67.1688889,4.85058824 68.4568889,5.24382353 69.6785185,5.78941176 L69.6785185,3.21794118 C68.507376,2.68868677 67.231762,2.42629483 65.9451852,2.45 C63.0228148,2.45 61.5108148,3.71205882 61.5108148,5.77705882" id="Shape"></path>
                        <polygon id="Path" points="102.152296 2.62088235 102.152296 13.8291176 104.558222 13.8291176 104.558222 5.28294118 105.572444 7.55176471 108.976 13.8291176 112 13.8291176 112 2.62088235 109.594074 2.62088235 109.594074 9.85558824 108.683556 7.75558824 105.949926 2.62088235"></polygon>
                        <rect id="Rectangle" x="84.2198519" y="2.62088235" width="2.62992593" height="11.2082353"></rect>
                        <path d="M81.1854815,10.5205882 C81.1854815,8.55852941 80.1380741,7.63823529 77.1804444,7.00617647 C75.5481481,6.64794118 75.152,6.28970588 75.152,5.77088235 C75.152,5.12235294 75.7368889,4.85058824 76.8112593,4.85058824 C78.1179259,4.85058824 79.4059259,5.24382353 80.6275556,5.78941176 L80.6275556,3.21794118 C79.4564131,2.68868677 78.180799,2.42629483 76.8942222,2.45 C73.9718519,2.45 72.4598519,3.71205882 72.4598519,5.77705882 C72.4598519,7.42411765 73.2334815,8.74588235 76.2574815,9.32441176 C78.0619259,9.69911765 78.4394074,9.98941176 78.4394074,10.5864706 C78.4394074,11.1835294 78.0619259,11.5417647 76.7801481,11.5417647 C75.2437916,11.5142162 73.7381877,11.1096383 72.3976296,10.3641176 L72.3976296,13.0611765 C73.3081481,13.5038235 74.5111111,14 76.7531852,14 C79.9161481,14 81.170963,12.6020588 81.170963,10.5205882" id="Path"></path>
                        <polygon id="Path" points="41.0998519 2.62088235 41.0998519 13.8291176 46.5048889 13.8291176 47.3552593 11.4079412 43.7463704 11.4079412 43.7463704 2.62088235"></polygon>
                        <polygon id="Path" points="30.4204444 2.62088235 30.4204444 5.04411765 33.3428148 5.04411765 33.3428148 13.8291176 35.9893333 13.8291176 35.9893333 5.04411765 39.117037 5.04411765 39.117037 2.62088235"></polygon>
                        <path d="M26.5834074,2.62088235 L23.1155556,2.62088235 L19.1748148,13.8291176 L22.1925926,13.8291176 L22.7442963,11.9411765 C24.1212835,12.3437114 25.5859758,12.3437114 26.962963,11.9411765 L27.5208889,13.8291176 L30.5282963,13.8291176 L26.5834074,2.62088235 Z M24.8494815,9.92147059 C24.3581003,9.92214608 23.869165,9.85281176 23.3976296,9.71558824 L24.8494815,4.82176471 L26.3013333,9.71558824 C25.829798,9.85281176 25.3408626,9.92214608 24.8494815,9.92147059 Z" id="Shape"></path>
                        <path d="M55.8216296,2.62088235 L52.3537778,2.62088235 L48.413037,13.8291176 L51.4204444,13.8291176 L51.9825185,11.9411765 C53.3569173,12.3420963 54.8184901,12.3420963 56.1928889,11.9411765 L56.7508148,13.8291176 L59.7582222,13.8291176 L55.8216296,2.62088235 Z M54.0877037,9.92147059 C53.5963226,9.92214608 53.1073872,9.85281176 52.6358519,9.71558824 L54.0877037,4.82176471 L55.5395556,9.71558824 C55.0680202,9.85281176 54.5790848,9.92214608 54.0877037,9.92147059 Z" id="Shape"></path>
                        <path d="M96.3241481,2.62088235 L92.8562963,2.62088235 L88.9155556,13.8291176 L91.922963,13.8291176 L92.485037,11.9411765 C93.8594358,12.3420963 95.3210087,12.3420963 96.6954074,11.9411765 L97.2533333,13.8291176 L100.260741,13.8291176 L96.3241481,2.62088235 Z M94.5902222,9.92147059 C94.0988411,9.92214608 93.6099057,9.85281176 93.1383704,9.71558824 L94.5902222,4.82176471 L96.0420741,9.71558824 C95.5705387,9.85281176 95.0816034,9.92214608 94.5902222,9.92147059 Z" id="Shape"></path>
                    </g>
                </g>
            </g>
        </g>
    </svg>
    """

public let mailIconSvg =
base64EncodedString <| """
    <svg height="16" viewBox="0 0 20 16" width="20" xmlns="http://www.w3.org/2000/svg"><g fill="none" fill-rule="evenodd" transform="translate(-2 -4)"><path d="m0 0h24v24h-24z"/><path d="m20 4h-16c-1.1 0-1.99.9-1.99 2l-.01 12c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2v-12c0-1.1-.9-2-2-2zm0 14h-16v-10l8 5 8-5zm-8-7-8-5h16z" fill="#121212" fill-rule="nonzero"/></g></svg>
    """
